<?php
/**
 * Emergency Access Page - صفحه دسترسی اضطراری
 * برای باز کردن محدودیت IP ادمین
 */

header('Content-Type: text/html; charset=utf-8');
mb_internal_encoding('UTF-8');
mb_http_output('UTF-8');

$message = '';
$error = '';
$success = false;

// Fixed admin email
$adminEmail = 'raytheme.ir@gmail.com';

// Start session for emergency access
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Include database configuration
require_once __DIR__ . '/config/database.php';
require_once __DIR__ . '/includes/functions.php';

// Database connection
function getDatabaseConnection() {
    try {
        // Use the same database configuration as the main application
        $db = new Database();
        $pdo = $db->getConnection();
        
        // Check the connection
        $pdo->query("SELECT 1");
        
        logSecurityEvent('DB_CONNECTION_SUCCESS', "Connected to database: " . $db->database . " on " . $db->host);
        return $pdo;
        
    } catch (PDOException $e) {
        logSecurityEvent('DB_CONNECTION_FAILED', "Database connection failed: " . $e->getMessage());
        error_log("Database connection failed: " . $e->getMessage());
        return null;
    } catch (Exception $e) {
        logSecurityEvent('DB_GENERAL_ERROR', "General database error: " . $e->getMessage());
        error_log("Database connection failed: " . $e->getMessage());
        return null;
    }
}

// Security: Get client IP
function getClientIP() {
    $ipKeys = ['HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_FORWARDED', 'HTTP_X_CLUSTER_CLIENT_IP', 'HTTP_FORWARDED_FOR', 'HTTP_FORWARDED', 'REMOTE_ADDR'];
    foreach ($ipKeys as $key) {
        if (array_key_exists($key, $_SERVER) === true) {
            foreach (explode(',', $_SERVER[$key]) as $ip) {
                $ip = trim($ip);
                if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) !== false) {
                    return $ip;
                }
            }
        }
    }
    return $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
}

// Security: Enhanced IP validation
function validateRequestIP($ip) {
    // Check if IP is from a known proxy/VPN service
    $suspiciousRanges = [
        '10.0.0.0/8', '172.16.0.0/12', '192.168.0.0/16', // Private ranges
        '127.0.0.0/8', // Localhost
    ];
    
    foreach ($suspiciousRanges as $range) {
        if (ipInRange($ip, $range)) {
            return false;
        }
    }
    
    return true;
}

// Security: Check if IP is in range
function ipInRange($ip, $range) {
    list($range, $netmask) = explode('/', $range, 2);
    $rangeDecimal = ip2long($range);
    $ipDecimal = ip2long($ip);
    $wildcardDecimal = pow(2, (32 - $netmask)) - 1;
    $netmaskDecimal = ~ $wildcardDecimal;
    
    return (($ipDecimal & $netmaskDecimal) == ($rangeDecimal & $netmaskDecimal));
}

// Security: Generate request fingerprint
function generateRequestFingerprint() {
    $data = [
        'ip' => getClientIP(),
        'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
        'accept_language' => $_SERVER['HTTP_ACCEPT_LANGUAGE'] ?? '',
        'accept_encoding' => $_SERVER['HTTP_ACCEPT_ENCODING'] ?? '',
        'timestamp' => time()
    ];
    
    return hash('sha256', json_encode($data) . 'GAVAHITO_SECRET_2024');
}

// Security: Enhanced rate limiting with fingerprint
function checkEnhancedRateLimit($ip, $fingerprint, $action = 'emergency_access', $maxAttempts = 3, $timeWindow = 3600) {
    $logFile = __DIR__ . '/emergency_access_log.txt';
    $currentTime = time();
    $attempts = [];
    
    if (file_exists($logFile)) {
        $lines = file($logFile, FILE_IGNORE_NEW_LINES);
        foreach ($lines as $line) {
            $data = explode('|', $line);
            if (count($data) >= 4 && $data[0] === $ip && $data[1] === $action) {
                $attemptTime = (int)$data[2];
                if ($currentTime - $attemptTime < $timeWindow) {
                    $attempts[] = $attemptTime;
                }
            }
        }
    }
    
    if (count($attempts) >= $maxAttempts) {
        logSecurityEvent('RATE_LIMIT_EXCEEDED_ENHANCED', "Rate limit exceeded for IP: $ip, Fingerprint: " . substr($fingerprint, 0, 10));
        return false;
    }
    
    // Log this attempt with fingerprint
    $logEntry = $ip . '|' . $action . '|' . $currentTime . '|' . $fingerprint . '|' . ($_SERVER['HTTP_USER_AGENT'] ?? 'Unknown') . "\n";
    file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
    
    return true;
}

// Security: Generate secure token and store in database
function generateSecureToken($email) {
    $token = bin2hex(random_bytes(32)); // 64 character hex string
    $expiresAt = time() + 3600; // 1 hour from now
    
    $pdo = getDatabaseConnection();
    if ($pdo) {
        try {
            // Clear old tokens for this email
            $stmt = $pdo->prepare("DELETE FROM admin_ip_tokens WHERE admin_email = ? OR expires_at < NOW()");
            $stmt->execute([$email]);
            
            // Insert new token
            $stmt = $pdo->prepare("INSERT INTO admin_ip_tokens (token, admin_email, expires_at) VALUES (?, ?, ?)");
            $result = $stmt->execute([$token, $email, date('Y-m-d H:i:s', $expiresAt)]);
            
            if ($result) {
                logSecurityEvent('TOKEN_GENERATED_SUCCESS', "Token generated in database for: $email");
                return $token;
            } else {
                logSecurityEvent('TOKEN_INSERT_FAILED', "Failed to insert token in database for: $email");
                return null;
            }
            
        } catch (PDOException $e) {
            logSecurityEvent('TOKEN_GENERATION_ERROR', "Database error: " . $e->getMessage());
            error_log("Token generation failed: " . $e->getMessage());
            return null;
        }
    } else {
        logSecurityEvent('DB_CONNECTION_FAILED', "Database connection failed for token generation");
        return null;
    }
}

// Security: Validate token from database
function validateSecureToken($token, $email) {
    $pdo = getDatabaseConnection();
    if ($pdo) {
        try {
            // Log the validation attempt
            logSecurityEvent('TOKEN_VALIDATION_ATTEMPT', "Attempting to validate token: " . substr($token, 0, 10) . "... for email: $email");
            
            // First, let's check if the token exists at all
            $stmt = $pdo->prepare("SELECT * FROM admin_ip_tokens WHERE token = ?");
            $stmt->execute([$token]);
            $tokenExists = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$tokenExists) {
                logSecurityEvent('TOKEN_NOT_FOUND', "Token not found in database");
                return false;
            }
            
            logSecurityEvent('TOKEN_EXISTS', "Token found - Email: {$tokenExists['admin_email']}, Used: {$tokenExists['is_used']}, Expires: {$tokenExists['expires_at']}");
            
            // Check if token is already used
            if ($tokenExists['is_used'] == 1) {
                logSecurityEvent('TOKEN_ALREADY_USED', "Token already used");
                return false;
            }
            
            // Check if token is expired
            $currentTime = date('Y-m-d H:i:s');
            if ($tokenExists['expires_at'] < $currentTime) {
                logSecurityEvent('TOKEN_EXPIRED', "Token expired - Current: $currentTime, Expires: {$tokenExists['expires_at']}");
                return false;
            }
            
            // Check if email matches
            if ($tokenExists['admin_email'] !== $email) {
                logSecurityEvent('TOKEN_EMAIL_MISMATCH', "Email mismatch - Expected: $email, Found: {$tokenExists['admin_email']}");
                return false;
            }
            
            // All checks passed, mark token as used
            $stmt = $pdo->prepare("UPDATE admin_ip_tokens SET is_used = 1 WHERE id = ?");
            $stmt->execute([$tokenExists['id']]);
            
            // Remove IP restrictions
            $stmt = $pdo->prepare("UPDATE admin_ip_restriction SET is_active = 0 WHERE admin_email = ?");
            $stmt->execute([$email]);
            
            logSecurityEvent('TOKEN_VALIDATED_DB', "Token validated successfully for: $email");
            return true;
            
        } catch (PDOException $e) {
            logSecurityEvent('TOKEN_VALIDATION_DB_ERROR', "Database validation error: " . $e->getMessage());
            return false;
        }
    } else {
        logSecurityEvent('DB_CONNECTION_FAILED_VALIDATION', "Database connection failed during validation");
        return false;
    }
}

// Security: Log security event
function logSecurityEvent($event, $details) {
    $logFile = __DIR__ . '/security_events.log';
    $timestamp = date('Y-m-d H:i:s');
    $ip = getClientIP();
    $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown';
    $logEntry = "[$timestamp] [$ip] [$event] $details | User-Agent: $userAgent\n";
    file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
}

// Email: Send emergency access email
function sendEmergencyEmail($toEmail, $token) {
    try {
        // Simple email sending without external dependencies
        $emergencyUrl = 'https://gavahito.ir/backend/Hua89fe754tr7gyu7o9werdhy79erwte0gw04etyg.php?token=' . $token;
        $clientIP = getClientIP();
        $timestamp = date('Y-m-d H:i:s');
        
        // Email headers
        $headers = array(
            'MIME-Version: 1.0',
            'Content-type: text/html; charset=UTF-8',
            'From: info@gavahito.ir',
            'Reply-To: info@gavahito.ir',
            'X-Mailer: PHP/' . phpversion()
        );
        
        // Email subject
        $subject = '🔓 دسترسی اضطراری به پنل ادمین - یادینو';
        
        // Email body
        $body = "
        <div style='font-family: Tahoma, Arial, sans-serif; direction: rtl; text-align: right;'>
            <h2 style='color: #e74c3c;'>🔓 دسترسی اضطراری به پنل ادمین</h2>
            <p>سلام،</p>
            <p>یک درخواست دسترسی اضطراری برای پنل ادمین ثبت شده است.</p>
            
            <div style='background: #f8f9fa; padding: 15px; border-radius: 8px; margin: 15px 0;'>
                <p><strong>جزئیات درخواست:</strong></p>
                <p>IP درخواست‌کننده: <code>$clientIP</code></p>
                <p>زمان درخواست: $timestamp</p>
            </div>
            
            <p><strong>توجه:</strong> این لینک فقط 1 ساعت معتبر است.</p>
            <div style='background: #f8f9fa; padding: 20px; border-radius: 10px; margin: 20px 0;'>
                <p><strong>لینک دسترسی اضطراری:</strong></p>
                <a href='$emergencyUrl' style='color: #3498db; text-decoration: none; font-weight: bold;'>
                    $emergencyUrl
                </a>
            </div>
            <p style='color: #e74c3c; font-weight: bold;'>⚠️ اگر شما این درخواست را نکرده‌اید، این ایمیل را نادیده بگیرید.</p>
            <hr>
            <p style='font-size: 12px; color: #7f8c8d;'>این ایمیل از سیستم امنیتی یادینو ارسال شده است.</p>
        </div>";
        
        // Send email using PHP's mail() function
        $result = mail($toEmail, $subject, $body, implode("\r\n", $headers));
        
        if ($result) {
            logSecurityEvent('EMAIL_SENT_SUCCESS', "Emergency email sent to: $toEmail");
            return true;
        } else {
            logSecurityEvent('EMAIL_SEND_FAILED', "mail() function failed for: $toEmail");
            return false;
        }
        
    } catch (Exception $e) {
        logSecurityEvent('EMAIL_SEND_ERROR', "Exception: " . $e->getMessage());
        return false;
    }
}

$clientIP = getClientIP();

// Handle token validation (for emergency access)
if (isset($_GET['token'])) {
    $token = urldecode($_GET['token'] ?? '');
    
    // Clean the token (remove any whitespace or special characters)
    $token = trim($token);
    
    // Log token validation attempt
    logSecurityEvent('EMERGENCY_TOKEN_VALIDATION', "Token: " . substr($token, 0, 10) . "... (Length: " . strlen($token) . ")");
    
    if (validateSecureToken($token, $adminEmail)) {
        $success = true;
        $message = '✅ محدودیت IP با موفقیت برداشته شد. حالا می‌توانید به پنل ادمین دسترسی داشته باشید.';
        logSecurityEvent('EMERGENCY_ACCESS_GRANTED', "IP restriction removed successfully");
        // Redirect to dashboard after successful validation
        header("Location: hf8fAS7wkd99iudu8.php");
        exit();
    } else {
        $error = '❌ توکن نامعتبر یا منقضی شده است.';
        logSecurityEvent('EMERGENCY_ACCESS_DENIED', "Invalid or expired token");
    }
}

// Handle emergency access request
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Validate IP first
    if (!validateRequestIP($clientIP)) {
        $error = '❌ دسترسی از این IP مجاز نیست.';
        logSecurityEvent('INVALID_IP_ACCESS', "Blocked access from invalid IP: $clientIP");
    } else {
        // Check rate limiting
        $fingerprint = generateRequestFingerprint();
        if (!checkEnhancedRateLimit($clientIP, $fingerprint, 'emergency_request', 3, 3600)) {
            $error = '❌ تعداد درخواست‌های شما بیش از حد مجاز است. لطفاً 1 ساعت دیگر تلاش کنید.';
            logSecurityEvent('RATE_LIMIT_EXCEEDED', "Emergency access rate limit exceeded for IP: $clientIP");
        } else {
            // Generate secure token
            $token = generateSecureToken($adminEmail);
            
            if ($token) {
                // Send email
                if (sendEmergencyEmail($adminEmail, $token)) {
                    $message = '📧 ایمیل دسترسی اضطراری به ' . $adminEmail . ' ارسال شد. لطفاً ایمیل خود را بررسی کنید.';
                    logSecurityEvent('EMERGENCY_EMAIL_SENT', "Emergency access email sent to: $adminEmail");
                } else {
                    $error = '❌ خطا در ارسال ایمیل. لطفاً دوباره تلاش کنید.';
                    logSecurityEvent('EMAIL_SEND_FAILED', "Failed to send emergency email to: $adminEmail");
                }
            } else {
                $error = '❌ خطا در تولید توکن. لطفاً دوباره تلاش کنید.';
                logSecurityEvent('TOKEN_GENERATION_FAILED', "Failed to generate token for: $adminEmail");
            }
        }
    }
}

// Security: Log page access
logSecurityEvent('EMERGENCY_PAGE_ACCESS', "Page accessed from IP: $clientIP");
?>
<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>دسترسی اضطراری - پنل ادمین یادینو</title>
    <link href="https://fonts.googleapis.com/css2?family=Vazirmatn:wght@300;400;500;600;700;800;900&display=swap" rel="stylesheet">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Vazirmatn', 'Tahoma', Arial, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            padding: 1rem;
        }
        
        .container {
            background: rgba(0, 0, 0, 0.4);
            border-radius: 2rem;
            backdrop-filter: blur(20px);
            border: 1px solid rgba(255, 255, 255, 0.1);
            box-shadow: 0 25px 50px rgba(0, 0, 0, 0.3);
            padding: 3rem 2rem;
            max-width: 500px;
            width: 100%;
            text-align: center;
        }
        
        .icon {
            font-size: 4rem;
            margin-bottom: 1.5rem;
            animation: pulse 2s infinite;
        }
        
        @keyframes pulse {
            0%, 100% { transform: scale(1); }
            50% { transform: scale(1.1); }
        }
        
        h1 {
            font-size: 2rem;
            margin-bottom: 1rem;
            color: #ffa726;
            font-weight: 700;
        }
        
        .description {
            font-size: 1.1rem;
            line-height: 1.6;
            margin-bottom: 2rem;
            opacity: 0.9;
        }
        
        .btn {
            background: linear-gradient(45deg, #4fc3f7, #29b6f6);
            color: white;
            border: none;
            padding: 1rem 2rem;
            border-radius: 0.8rem;
            font-size: 1rem;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            font-family: inherit;
            text-decoration: none;
            display: inline-block;
        }
        
        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 20px rgba(0, 0, 0, 0.2);
        }
        
        .alert {
            padding: 1rem;
            border-radius: 0.8rem;
            margin-bottom: 1.5rem;
            font-weight: 500;
        }
        
        .alert-success {
            background: rgba(76, 175, 80, 0.2);
            border: 1px solid rgba(76, 175, 80, 0.3);
            color: #4caf50;
        }
        
        .alert-danger {
            background: rgba(244, 67, 54, 0.2);
            border: 1px solid rgba(244, 67, 54, 0.3);
            color: #f44336;
        }
        
        .success-actions {
            margin-top: 2rem;
            padding-top: 1.5rem;
            border-top: 1px solid rgba(255, 255, 255, 0.2);
        }
        
        .success-actions a {
            display: inline-block;
            background: linear-gradient(45deg, #4caf50, #66bb6a);
            color: white;
            text-decoration: none;
            padding: 0.8rem 1.5rem;
            border-radius: 0.8rem;
            margin: 0.5rem;
            font-weight: 500;
            transition: all 0.3s ease;
        }
        
        .success-actions a:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 16px rgba(0, 0, 0, 0.2);
        }
        
        .warning {
            background: rgba(255, 152, 0, 0.2);
            border: 1px solid rgba(255, 152, 0, 0.3);
            color: #ff9800;
            padding: 1rem;
            border-radius: 0.8rem;
            margin-bottom: 1.5rem;
            font-size: 0.9rem;
        }
        
        .email-info {
            background: rgba(255, 255, 255, 0.1);
            border: 1px solid rgba(255, 255, 255, 0.2);
            padding: 1rem;
            border-radius: 0.8rem;
            margin: 1rem 0;
            font-family: monospace;
            font-size: 0.9rem;
        }
        
        .security-info {
            background: rgba(255, 255, 255, 0.05);
            border: 1px solid rgba(255, 255, 255, 0.1);
            padding: 0.8rem;
            border-radius: 0.5rem;
            margin: 1rem 0;
            font-size: 0.8rem;
            opacity: 0.7;
        }
        
        @media (max-width: 768px) {
            .container {
                padding: 2rem 1rem;
                margin: 1rem;
            }
            
            h1 {
                font-size: 1.5rem;
            }
            
            .icon {
                font-size: 3rem;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <?php if ($success): ?>
            <div class="icon">✅</div>
            <h1>دسترسی اضطراری فعال شد</h1>
            <div class="description">
                محدودیت IP با موفقیت برداشته شد و حالا می‌توانید به پنل ادمین دسترسی داشته باشید.
            </div>
            
            <div class="alert alert-success">
                <?php echo htmlspecialchars($message); ?>
            </div>
            
            <div class="success-actions">
                <a href="hf8fAS7wkd99iudu8.php" class="btn" style="background: linear-gradient(45deg, #4caf50, #66bb6a);">
                    ورود به پنل ادمین
                </a>
                <a href="security.php" class="btn" style="background: linear-gradient(45deg, #ff9800, #ffb74d);">
                    مدیریت امنیت
                </a>
            </div>
            
            <script>
                // Auto redirect to dashboard after 3 seconds
                setTimeout(function() {
                    window.location.href = 'hf8fAS7wkd99iudu8.php';
                }, 3000);
            </script>
            
        <?php else: ?>
            <div class="icon">🔓</div>
            <h1>دسترسی اضطراری</h1>
            <div class="description">
                برای باز کردن محدودیت IP و دسترسی به پنل ادمین، روی دکمه زیر کلیک کنید.
            </div>
            
            <div class="warning">
                ⚠️ این عملیات فقط در مواقع اضطراری استفاده شود. لینک دسترسی اضطراری به ایمیل ادمین ارسال خواهد شد.
            </div>
            
            <div class="email-info">
                ایمیل ادمین: <?php echo htmlspecialchars($adminEmail); ?>
            </div>
            
            <?php if ($error): ?>
                <div class="alert alert-danger">
                    <?php echo htmlspecialchars($error); ?>
                </div>
            <?php endif; ?>
            
            <?php if ($message): ?>
                <div class="alert alert-success">
                    <?php echo htmlspecialchars($message); ?>
                </div>
            <?php endif; ?>
            
            <form method="POST" action="">
                <button type="submit" class="btn">
                    ارسال لینک دسترسی اضطراری
                </button>
            </form>
            
            <div class="security-info">
                🔒 تمام فعالیت‌ها ثبت و نظارت می‌شود
            </div>
        <?php endif; ?>
    </div>
</body>
</html>
