<?php
/**
 * Simple API Endpoint for Form Submissions
 * API ساده برای ارسال فرم‌های گزارش مشکل و درخواست ویژگی
 */

// Error reporting for production
error_reporting(0);
ini_set('display_errors', 0);

// Enable CORS
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
header('Content-Type: application/json; charset=utf-8');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit();
}

// Start output buffering to catch any unexpected output
ob_start();

try {
    // Include required files
    require_once '../config/database.php';
    require_once '../includes/functions.php';
    require_once '../config/email-config.php';
    require_once '../config/recaptcha-config.php';
    
    // Get JSON input
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        throw new Exception('Invalid JSON input');
    }
    
    // Validate required fields
    $requiredFields = ['form_type', 'name', 'email', 'subject', 'description', 'priority'];
    foreach ($requiredFields as $field) {
        if (empty($input[$field])) {
            throw new Exception("Field '$field' is required");
        }
    }
    
    // Sanitize input with enhanced security
    $formType = preg_replace('/[^a-zA-Z_]/', '', $input['form_type']);
    $name = preg_replace('/[^a-zA-Z0-9\s\-_\.]/', '', $input['name']);
    $email = filter_var($input['email'], FILTER_SANITIZE_EMAIL);
    $subject = preg_replace('/[<>]/', '', $input['subject']);
    $description = preg_replace('/[<>]/', '', $input['description']);
    $priority = preg_replace('/[^a-zA-Z]/', '', $input['priority']);
    
    // Validate form type
    if (!in_array($formType, ['bug_report', 'feature_request'])) {
        throw new Exception('Invalid form type');
    }
    
    // Validate email
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        throw new Exception('Invalid email format');
    }
    
    // Validate priority
    $validPriorities = ['low', 'medium', 'high', 'critical'];
    if (!in_array($priority, $validPriorities)) {
        throw new Exception('Invalid priority level');
    }
    
    // Get client IP
    $clientIP = getClientIP();
    
    // Verify reCAPTCHA
    if (isset($input['recaptcha_token'])) {
        $recaptchaResult = verifyRecaptcha($input['recaptcha_token']);
        if (!$recaptchaResult['success']) {
            throw new Exception($recaptchaResult['message']);
        }
    }
    
    // Database operations
    $db = new Database();
    $conn = $db->getConnection();
    
    // Insert into database
    if ($formType === 'bug_report') {
        $insertQuery = "INSERT INTO bug_reports (name, email, subject, description, priority, status, ip_address, user_agent) 
                       VALUES (?, ?, ?, ?, ?, 'pending', ?, ?)";
    } else {
        $insertQuery = "INSERT INTO feature_requests (name, email, feature, description, priority, status, ip_address, user_agent) 
                       VALUES (?, ?, ?, ?, ?, 'pending', ?, ?)";
    }
    
    $stmt = $conn->prepare($insertQuery);
    $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown';
    $stmt->execute([$name, $email, $subject, $description, $priority, $clientIP, $userAgent]);
    
    $submissionId = $conn->lastInsertId();
    
    // Send email notification (optional)
    $emailSent = false;
    $userEmailSent = false;
    
    try {
        // Get admin email from settings
        $settingsQuery = "SELECT setting_value FROM system_settings WHERE setting_key = 'admin_email'";
        $settingsStmt = $conn->query($settingsQuery);
        $adminEmail = $settingsStmt->fetchColumn() ?: 'info@gavahito.ir';
        
        // Send to admin using improved function
        $emailData = [
            'name' => $name,
            'email' => $email,
            'subject' => $subject,
            'priority' => $priority,
            'description' => $description,
            'ip' => $clientIP
        ];
        
        if ($formType === 'feature_request') {
            $emailData['feature'] = $subject;
        }
        
        $emailSent = sendNotificationEmail($formType, $emailData);
        
        // Send to user
        $userSubject = $formType === 'bug_report' ? '✅ تایید گزارش مشکل' : '✅ تایید درخواست ویژگی';
        $userBody = '
        <h3 style="color: #27ae60; margin-bottom: 20px;">✅ ' . ($formType === 'bug_report' ? 'گزارش مشکل' : 'درخواست ویژگی') . ' شما با موفقیت دریافت شد</h3>
        
        <div class="info-row">
            <span class="info-label">👤 نام:</span>
            <span class="info-value">' . htmlspecialchars($name) . '</span>
        </div>
        
        <div class="info-row">
            <span class="info-label">📝 موضوع:</span>
            <span class="info-value">' . htmlspecialchars($subject) . '</span>
        </div>
        
        <div class="info-row">
            <span class="info-label">⚡ اولویت:</span>
            <span class="info-value">
                <span class="priority-badge" style="background-color: ' . getPriorityColor($priority) . '">' . getPriorityLabel($priority) . '</span>
            </span>
        </div>
        
        <div class="info-row">
            <span class="info-label">🔢 شماره پیگیری:</span>
            <span class="info-value">' . $submissionId . '</span>
        </div>
        
        <div style="margin-top: 20px; padding: 15px; background: #e8f5e8; border: 1px solid #4caf50; border-radius: 8px;">
            <p style="margin: 0; color: #2e7d32; font-weight: 600;">📞 ما به زودی با شما تماس خواهیم گرفت</p>
        </div>
        
        <div style="margin-top: 20px; padding: 15px; background: #fff3cd; border: 1px solid #ffc107; border-radius: 8px;">
            <p style="margin: 0; color: #856404;">💡 برای پیگیری بیشتر می‌توانید از شماره پیگیری بالا استفاده کنید</p>
        </div>';
        
        $userEmailSent = sendEmail($email, $userSubject, $userBody);
        
    } catch (Exception $emailError) {
        // Email errors should not break the form submission
        error_log("Email error: " . $emailError->getMessage());
    }
    
    // Clear any previous output
    ob_clean();
    
    // Return success response
    echo json_encode([
        'success' => true,
        'message' => $formType === 'bug_report' ? 'گزارش مشکل با موفقیت ارسال شد' : 'درخواست ویژگی با موفقیت ارسال شد',
        'submission_id' => $submissionId,
        'email_sent' => $emailSent,
        'user_email_sent' => $userEmailSent
    ]);
    
} catch (Exception $e) {
    // Clear any previous output
    ob_clean();
    
    error_log("Form submission error: " . $e->getMessage());
    
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
?>
