<?php
/**
 * IP Check API - API بررسی IP برای لندینگ
 * این API از سیستم واقعی دیتابیس استفاده می‌کند
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Handle OPTIONS request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Include required files
require_once '../../config/database.php';
require_once '../../includes/functions.php';

try {
    $input = json_decode(file_get_contents('php://input'), true);
    $action = $input['action'] ?? '';
    
    if ($action === 'check_ip') {
        $clientIP = getClientIP();
        
        // Check if IP is whitelisted first
        if (isIPWhitelisted($clientIP)) {
            echo json_encode([
                'blocked' => false,
                'ip' => $clientIP,
                'message' => 'IP در لیست سفید قرار دارد'
            ]);
            exit();
        }
        
        // Check if IP is blacklisted
        $isBlocked = isIPBlacklisted($clientIP);
        
        if ($isBlocked) {
            // Get block details from database
            $blockDetails = getBlockDetails($clientIP);
            
            // Log the blocked access attempt
            logSecurityEvent('landing_page_blocked_access', 'Blocked IP tried to access landing page', 'medium', json_encode([
                'ip' => $clientIP,
                'user_agent' => getUserAgent(),
                'current_url' => $input['current_url'] ?? '',
                'referer' => $_SERVER['HTTP_REFERER'] ?? ''
            ]));
            
            echo json_encode([
                'blocked' => true,
                'ip' => $clientIP,
                'reason' => $blockDetails['reason'] ?? 'IP مسدود شده',
                'expires_at' => $blockDetails['expires_at'] ? date('Y/m/d H:i', strtotime($blockDetails['expires_at'])) : null,
                'time_remaining' => $blockDetails['expires_at'] ? getTimeRemaining($blockDetails['expires_at']) : null
            ]);
        } else {
            // Log page access for monitoring
            logSecurityEvent('landing_page_access', 'Landing page accessed', 'low', json_encode([
                'ip' => $clientIP,
                'user_agent' => getUserAgent(),
                'current_url' => $input['current_url'] ?? ''
            ]));
            
            echo json_encode([
                'blocked' => false,
                'ip' => $clientIP,
                'message' => 'دسترسی مجاز'
            ]);
        }
    } else {
        echo json_encode([
            'error' => 'Invalid action',
            'message' => 'لطفاً action را به درستی تنظیم کنید'
        ]);
    }
    
} catch (Exception $e) {
    error_log("IP check API error: " . $e->getMessage());
    echo json_encode([
        'error' => 'Internal server error',
        'message' => $e->getMessage(),
        'blocked' => false // Allow access if error occurs
    ]);
}

// Helper functions
function getBlockDetails($ip) {
    try {
        $db = new Database();
        $conn = $db->getConnection();
        
        $query = "SELECT reason, expires_at, created_at FROM ip_blacklist WHERE ip_address = ? AND (expires_at IS NULL OR expires_at > NOW()) ORDER BY created_at DESC LIMIT 1";
        $stmt = $conn->prepare($query);
        $stmt->execute([$ip]);
        
        return $stmt->fetch();
    } catch (Exception $e) {
        error_log("Block details error: " . $e->getMessage());
        return null;
    }
}

function getTimeRemaining($expiresAt) {
    $expires = strtotime($expiresAt);
    $now = time();
    $diff = $expires - $now;
    
    if ($diff <= 0) {
        return 'منقضی شده';
    }
    
    $hours = floor($diff / 3600);
    $minutes = floor(($diff % 3600) / 60);
    
    if ($hours > 0) {
        return "{$hours} ساعت و {$minutes} دقیقه باقی مانده";
    } else {
        return "{$minutes} دقیقه باقی مانده";
    }
}
?>
