<?php
// Set proper charset headers
header('Content-Type: text/html; charset=utf-8');
mb_internal_encoding('UTF-8');
mb_http_output('UTF-8');

// Start session
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once 'config/database.php';
require_once 'includes/functions.php';

// Check authentication
if (!isLoggedIn()) {
    logSecurityEvent('unauthorized_access_attempt', 'Unauthorized access attempt to bug-reports.php from IP: ' . getClientIP(), 'high');
    header('Location: 403.php');
    exit;
}
checkSessionTimeout();

$message = '';
$error = '';

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    try {
        $db = new Database();
        $conn = $db->getConnection();
        
        switch ($action) {
            case 'update_status':
                $bugId = (int)($_POST['bug_id'] ?? 0);
                $status = sanitizeInput($_POST['status'] ?? '');
                $priority = sanitizeInput($_POST['priority'] ?? '');
                
                if ($bugId && $status) {
                    $updateQuery = "UPDATE bug_reports SET status = ?, priority = ?, updated_at = NOW() WHERE id = ?";
                    $updateStmt = $conn->prepare($updateQuery);
                    $updateStmt->execute([$status, $priority, $bugId]);
                    
                    $message = "وضعیت گزارش باگ با موفقیت بروزرسانی شد";
                    logSecurityEvent('bug_status_updated', 'Bug ID: ' . $bugId . ', Status: ' . $status, 'low');
                }
                break;
                
            case 'bulk_update':
                $selectedBugs = $_POST['selected_bugs'] ?? [];
                $bulkStatus = sanitizeInput($_POST['bulk_status'] ?? '');
                $bulkPriority = sanitizeInput($_POST['bulk_priority'] ?? '');
                
                if (!empty($selectedBugs) && $bulkStatus) {
                    $updateQuery = "UPDATE bug_reports SET status = ?, priority = ?, updated_at = NOW() WHERE id IN (" . str_repeat('?,', count($selectedBugs) - 1) . "?)";
                    $params = array_merge([$bulkStatus, $bulkPriority], $selectedBugs);
                    $updateStmt = $conn->prepare($updateQuery);
                    $updateStmt->execute($params);
                    
                    $message = count($selectedBugs) . " گزارش باگ با موفقیت بروزرسانی شد";
                    logSecurityEvent('bugs_bulk_updated', 'Count: ' . count($selectedBugs), 'low');
                }
                break;
                
            case 'delete':
                $bugId = (int)($_POST['bug_id'] ?? 0);
                
                if ($bugId) {
                    $deleteQuery = "DELETE FROM bug_reports WHERE id = ?";
                    $deleteStmt = $conn->prepare($deleteQuery);
                    $deleteStmt->execute([$bugId]);
                    
                    $message = "گزارش باگ با موفقیت حذف شد";
                    logSecurityEvent('bug_deleted', 'Bug ID: ' . $bugId, 'medium');
                }
                break;
                
            case 'bulk_delete':
                $selectedBugs = $_POST['selected_bugs'] ?? [];
                
                if (!empty($selectedBugs)) {
                    $deleteQuery = "DELETE FROM bug_reports WHERE id IN (" . str_repeat('?,', count($selectedBugs) - 1) . "?)";
                    $deleteStmt = $conn->prepare($deleteQuery);
                    $deleteStmt->execute($selectedBugs);
                    
                    $message = count($selectedBugs) . " گزارش باگ با موفقیت حذف شد";
                    logSecurityEvent('bugs_bulk_deleted', 'Count: ' . count($selectedBugs), 'medium');
                }
                break;
                
            case 'get_bug_details':
                $bugId = (int)($_POST['bug_id'] ?? 0);
                
                if ($bugId) {
                    $bugQuery = "SELECT * FROM bug_reports WHERE id = ?";
                    $bugStmt = $conn->prepare($bugQuery);
                    $bugStmt->execute([$bugId]);
                    $bug = $bugStmt->fetch();
                    
                    if ($bug) {
                        header('Content-Type: application/json');
                        echo json_encode([
                            'success' => true,
                            'bug' => [
                                'id' => $bug['id'],
                                'subject' => htmlspecialchars($bug['subject']),
                                'description' => htmlspecialchars($bug['description']),
                                'name' => htmlspecialchars($bug['name']),
                                'email' => htmlspecialchars($bug['email']),
                                'status' => $bug['status'],
                                'priority' => $bug['priority'],
                                'created_at' => $bug['created_at']
                            ]
                        ]);
                    } else {
                        header('Content-Type: application/json');
                        echo json_encode(['success' => false, 'message' => 'گزارش باگ یافت نشد']);
                    }
                } else {
                    header('Content-Type: application/json');
                    echo json_encode(['success' => false, 'message' => 'شناسه گزارش نامعتبر است']);
                }
                exit;
                break;
        }
    } catch (Exception $e) {
        error_log("Bug reports error: " . $e->getMessage());
        $error = 'خطا در عملیات. لطفاً دوباره تلاش کنید';
    }
}

// Get bug reports with filtering
$statusFilter = $_GET['status'] ?? '';
$priorityFilter = $_GET['priority'] ?? '';
$searchQuery = $_GET['search'] ?? '';

try {
    $db = new Database();
    $conn = $db->getConnection();
    
    $whereConditions = [];
    $params = [];
    
    if ($statusFilter) {
        $whereConditions[] = "status = ?";
        $params[] = $statusFilter;
    }
    
    if ($priorityFilter) {
        $whereConditions[] = "priority = ?";
        $params[] = $priorityFilter;
    }
    
    if ($searchQuery) {
        $whereConditions[] = "(subject LIKE ? OR description LIKE ? OR name LIKE ? OR email LIKE ?)";
        $searchParam = '%' . $searchQuery . '%';
        $params[] = $searchParam;
        $params[] = $searchParam;
        $params[] = $searchParam;
        $params[] = $searchParam;
    }
    
    $whereClause = '';
    if (!empty($whereConditions)) {
        $whereClause = 'WHERE ' . implode(' AND ', $whereConditions);
    }
    
    $bugsQuery = "SELECT * FROM bug_reports $whereClause ORDER BY created_at DESC";
    $bugsStmt = $conn->prepare($bugsQuery);
    $bugsStmt->execute($params);
    $bugs = $bugsStmt->fetchAll();
    
    // Get statistics
    $statsQuery = "SELECT 
        COUNT(*) as total,
        SUM(CASE WHEN status = 'new' THEN 1 ELSE 0 END) as new_count,
        SUM(CASE WHEN status = 'in_progress' THEN 1 ELSE 0 END) as in_progress_count,
        SUM(CASE WHEN status = 'resolved' THEN 1 ELSE 0 END) as resolved_count,
        SUM(CASE WHEN status = 'closed' THEN 1 ELSE 0 END) as closed_count
        FROM bug_reports";
    $statsStmt = $conn->query($statsQuery);
    $stats = $statsStmt->fetch();
    
} catch (Exception $e) {
    error_log("Bug reports list error: " . $e->getMessage());
    $error = 'خطا در بارگذاری گزارش‌های باگ';
    $bugs = [];
    $stats = ['total' => 0, 'new_count' => 0, 'in_progress_count' => 0, 'resolved_count' => 0, 'closed_count' => 0];
}
?>
<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>گزارش‌های باگ - پنل مدیریت گواهیتو</title>
    <link rel="stylesheet" href="assets/admin-style.css">
    <link rel="icon" type="image/png" href="assets/images/logo.png">
    <?php include 'includes/admin-styles.php'; ?>
</head>
<body>
    <?php include 'includes/header.php'; ?>
    <?php include 'includes/sidebar.php'; ?>
    
    <!-- Sidebar Overlay for Mobile -->
    <div class="sidebar-overlay" id="sidebarOverlay"></div>

    <!-- Main Content -->
    <div class="main-content" id="mainContent">
        <div class="page-header">
            <h1 class="page-title">🐛 گزارش‌های باگ</h1>
            <p class="page-description">مدیریت و پیگیری گزارش‌های باگ سیستم</p>
        </div>

        <?php if ($error): ?>
            <div class="alert alert-danger">
                <?php echo htmlspecialchars($error); ?>
            </div>
        <?php endif; ?>

        <?php if ($message): ?>
            <div class="alert alert-success">
                <?php echo htmlspecialchars($message); ?>
            </div>
        <?php endif; ?>

        <!-- Statistics -->
        <div class="grid grid-cols-1 grid-cols-2 grid-cols-3 grid-cols-4" style="margin-bottom: 2rem;">
            <div class="stats-card">
                <div class="stats-number"><?php echo $stats['total']; ?></div>
                <div class="stats-label">کل گزارش‌ها</div>
            </div>
            
            <div class="stats-card">
                <div class="stats-number"><?php echo $stats['new_count']; ?></div>
                <div class="stats-label">جدید</div>
            </div>
            
            <div class="stats-card">
                <div class="stats-number"><?php echo $stats['in_progress_count']; ?></div>
                <div class="stats-label">در حال بررسی</div>
            </div>
            
            <div class="stats-card">
                <div class="stats-number"><?php echo $stats['resolved_count']; ?></div>
                <div class="stats-label">حل شده</div>
            </div>
            
            <div class="stats-card">
                <div class="stats-number"><?php echo $stats['closed_count']; ?></div>
                <div class="stats-label">بسته</div>
            </div>
        </div>

        <!-- Filters -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">🔍 فیلترها</h2>
            </div>
            
            <form method="GET" action="" style="display: flex; gap: 1rem; align-items: end; flex-wrap: wrap;">
                <div class="form-group" style="flex: 1; min-width: 200px;">
                    <label for="search" class="form-label">جستجو</label>
                    <input type="text" id="search" name="search" class="form-input" 
                           value="<?php echo htmlspecialchars($searchQuery); ?>" 
                           placeholder="جستجو در موضوع، توضیحات، نام یا ایمیل">
                </div>
                
                <div class="form-group" style="min-width: 150px;">
                    <label for="status" class="form-label">وضعیت</label>
                    <select id="status" name="status" class="form-select">
                        <option value="">همه</option>
                        <option value="new" <?php echo $statusFilter === 'new' ? 'selected' : ''; ?>>جدید</option>
                        <option value="in_progress" <?php echo $statusFilter === 'in_progress' ? 'selected' : ''; ?>>در حال بررسی</option>
                        <option value="resolved" <?php echo $statusFilter === 'resolved' ? 'selected' : ''; ?>>حل شده</option>
                        <option value="closed" <?php echo $statusFilter === 'closed' ? 'selected' : ''; ?>>بسته</option>
                    </select>
                </div>
                
                <div class="form-group" style="min-width: 150px;">
                    <label for="priority" class="form-label">اولویت</label>
                    <select id="priority" name="priority" class="form-select">
                        <option value="">همه</option>
                        <option value="low" <?php echo $priorityFilter === 'low' ? 'selected' : ''; ?>>کم</option>
                        <option value="medium" <?php echo $priorityFilter === 'medium' ? 'selected' : ''; ?>>متوسط</option>
                        <option value="high" <?php echo $priorityFilter === 'high' ? 'selected' : ''; ?>>زیاد</option>
                        <option value="critical" <?php echo $priorityFilter === 'critical' ? 'selected' : ''; ?>>بحرانی</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <button type="submit" class="btn btn-primary">اعمال فیلتر</button>
                    <a href="bug-reports.php" class="btn btn-secondary">پاک کردن</a>
                </div>
            </form>
        </div>

        <!-- Bug Reports List -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">📋 لیست گزارش‌های باگ</h2>
                <div style="display: flex; gap: 0.5rem;">
                    <button type="button" class="btn btn-warning btn-sm" onclick="showBulkUpdateModal()">
                        بروزرسانی دسته‌ای
                    </button>
                    <button type="button" class="btn btn-danger btn-sm" onclick="deleteSelectedBugs()">
                        حذف انتخاب شده‌ها
                    </button>
                </div>
            </div>
            
            <?php if (!empty($bugs)): ?>
                <form method="POST" action="" id="bugsForm">
                    <input type="hidden" name="action" value="bulk_delete">
                    
                    <div class="table-container">
                        <table class="table">
                            <thead>
                                <tr>
                                    <th>
                                        <input type="checkbox" id="selectAll" onchange="toggleSelectAll()">
                                    </th>
                                    <th>موضوع</th>
                                    <th>گزارش‌دهنده</th>
                                    <th>اولویت</th>
                                    <th>وضعیت</th>
                                    <th>تاریخ</th>
                                    <th>عملیات</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($bugs as $bug): ?>
                                    <tr>
                                        <td>
                                            <input type="checkbox" name="selected_bugs[]" value="<?php echo $bug['id']; ?>" class="bug-checkbox">
                                        </td>
                                        <td>
                                            <strong><?php echo htmlspecialchars($bug['subject']); ?></strong>
                                            <br>
                                            <small style="color: var(--dark-400);">
                                                <?php echo htmlspecialchars(substr($bug['description'], 0, 100)) . (strlen($bug['description']) > 100 ? '...' : ''); ?>
                                            </small>
                                        </td>
                                        <td>
                                            <div><?php echo htmlspecialchars($bug['name']); ?></div>
                                            <small style="color: var(--dark-400);"><?php echo htmlspecialchars($bug['email']); ?></small>
                                        </td>
                                        <td><?php echo getPriorityBadge($bug['priority']); ?></td>
                                        <td><?php echo getStatusBadge($bug['status']); ?></td>
                                        <td><?php echo formatDate($bug['created_at']); ?></td>
                                        <td>
                                            <div style="display: flex; gap: 0.5rem; flex-wrap: wrap;">
                                                <button type="button" class="btn btn-warning btn-sm" onclick="editBug(<?php echo $bug['id']; ?>, '<?php echo htmlspecialchars($bug['subject']); ?>', '<?php echo $bug['status']; ?>', '<?php echo $bug['priority']; ?>')">
                                                    ویرایش
                                                </button>
                                                <button type="button" class="btn btn-info btn-sm" onclick="viewBugAjax(<?php echo $bug['id']; ?>)">
                                                    مشاهده
                                                </button>
                                                <button type="button" class="btn btn-danger btn-sm" onclick="deleteBug(<?php echo $bug['id']; ?>)">
                                                    حذف
                                                </button>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </form>
            <?php else: ?>
                <p style="color: var(--dark-400); text-align: center; padding: 2rem;">هیچ گزارش باگی یافت نشد</p>
            <?php endif; ?>
        </div>
    </div>

    <!-- Edit Bug Modal -->
    <div id="editModal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); z-index: 1000;">
        <div style="position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%); background: var(--dark-800); padding: 2rem; border-radius: 1rem; min-width: 400px;">
            <h3 style="margin-bottom: 1rem; color: var(--primary-400);">ویرایش گزارش باگ</h3>
            
            <form method="POST" action="" id="editBugForm">
                <input type="hidden" name="action" value="update_status">
                <input type="hidden" name="bug_id" id="editBugId">
                
                <div class="form-group">
                    <label for="editSubject" class="form-label">موضوع</label>
                    <input type="text" id="editSubject" class="form-input" readonly>
                </div>
                
                <div class="form-group">
                    <label for="editStatus" class="form-label">وضعیت</label>
                    <select id="editStatus" name="status" class="form-select">
                        <option value="new">جدید</option>
                        <option value="in_progress">در حال بررسی</option>
                        <option value="resolved">حل شده</option>
                        <option value="closed">بسته</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label for="editPriority" class="form-label">اولویت</label>
                    <select id="editPriority" name="priority" class="form-select">
                        <option value="low">کم</option>
                        <option value="medium">متوسط</option>
                        <option value="high">زیاد</option>
                        <option value="critical">بحرانی</option>
                    </select>
                </div>
                
                <div style="display: flex; gap: 1rem; margin-top: 1rem;">
                    <button type="submit" class="btn btn-primary">ذخیره تغییرات</button>
                    <button type="button" class="btn btn-secondary" onclick="closeEditModal()">انصراف</button>
                </div>
            </form>
        </div>
    </div>

    <!-- View Bug Modal -->
    <div id="viewModal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); z-index: 1000;">
        <div style="position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%); background: var(--dark-800); padding: 2rem; border-radius: 1rem; min-width: 500px; max-width: 90%; max-height: 90%; display: flex; flex-direction: column; box-shadow: 0 20px 40px rgba(0,0,0,0.3);">
            <h3 style="margin-bottom: 1rem; color: var(--primary-400); flex-shrink: 0;">مشاهده گزارش باگ</h3>
            
            <div id="viewBugContent" style="flex: 1; overflow-y: auto; padding-right: 0.5rem; scrollbar-width: thin; scrollbar-color: var(--dark-600) var(--dark-800);">
                <!-- Content will be filled by JavaScript -->
            </div>
            
            <div style="display: flex; gap: 1rem; margin-top: 1rem; flex-shrink: 0;">
                <button type="button" class="btn btn-secondary" onclick="closeViewModal()">بستن</button>
            </div>
        </div>
    </div>

    <!-- Bulk Update Modal -->
    <div id="bulkUpdateModal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); z-index: 1000;">
        <div style="position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%); background: var(--dark-800); padding: 2rem; border-radius: 1rem; min-width: 400px;">
            <h3 style="margin-bottom: 1rem; color: var(--primary-400);">بروزرسانی دسته‌ای</h3>
            <p id="bulkUpdateCount" style="margin-bottom: 1rem; color: var(--dark-300);"></p>
            
            <form method="POST" action="" id="bulkUpdateForm">
                <input type="hidden" name="action" value="bulk_update">
                <div id="bulkUpdateSelectedBugs">
                    <!-- Selected bug IDs will be added here dynamically -->
                </div>
                
                <div class="form-group">
                    <label for="bulkStatus" class="form-label">وضعیت جدید</label>
                    <select id="bulkStatus" name="bulk_status" class="form-select">
                        <option value="">انتخاب کنید</option>
                        <option value="new">جدید</option>
                        <option value="in_progress">در حال بررسی</option>
                        <option value="resolved">حل شده</option>
                        <option value="closed">بسته</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label for="bulkPriority" class="form-label">اولویت جدید</label>
                    <select id="bulkPriority" name="bulk_priority" class="form-select">
                        <option value="">انتخاب کنید</option>
                        <option value="low">کم</option>
                        <option value="medium">متوسط</option>
                        <option value="high">زیاد</option>
                        <option value="critical">بحرانی</option>
                    </select>
                </div>
                
                <div style="display: flex; gap: 1rem; margin-top: 1rem;">
                    <button type="submit" class="btn btn-primary">بروزرسانی</button>
                    <button type="button" class="btn btn-secondary" onclick="closeBulkUpdateModal()">انصراف</button>
                </div>
            </form>
        </div>
    </div>

    <script>
        // Toggle select all checkbox
        function toggleSelectAll() {
            const selectAll = document.getElementById('selectAll');
            const checkboxes = document.querySelectorAll('.bug-checkbox');
            
            checkboxes.forEach(checkbox => {
                checkbox.checked = selectAll.checked;
            });
        }
        
        // Delete selected bugs
        function deleteSelectedBugs() {
            const checkboxes = document.querySelectorAll('.bug-checkbox:checked');
            
            if (checkboxes.length === 0) {
                alert('لطفاً حداقل یک گزارش را انتخاب کنید');
                return;
            }
            
            if (confirm('آیا از حذف ' + checkboxes.length + ' گزارش انتخاب شده اطمینان دارید؟')) {
                document.getElementById('bugsForm').submit();
            }
        }
        
        // Edit bug modal
        function editBug(bugId, subject, status, priority) {
            document.getElementById('editBugId').value = bugId;
            document.getElementById('editSubject').value = subject;
            document.getElementById('editStatus').value = status;
            document.getElementById('editPriority').value = priority;
            document.getElementById('editModal').style.display = 'block';
        }
        
        function closeEditModal() {
            document.getElementById('editModal').style.display = 'none';
        }
        
        // View bug modal using AJAX
        function viewBugAjax(bugId) {
            // Show loading state
            document.getElementById('viewBugContent').innerHTML = '<div style="text-align: center; padding: 2rem; color: var(--dark-300);">در حال بارگذاری...</div>';
            document.getElementById('viewModal').style.display = 'block';
            
            // Create form data
            const formData = new FormData();
            formData.append('action', 'get_bug_details');
            formData.append('bug_id', bugId);
            
            // Send AJAX request
            fetch('bug-reports.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.text())
            .then(data => {
                try {
                    const bugData = JSON.parse(data);
                    if (bugData.success) {
                        const content = `
                            <div class="form-group">
                                <label class="form-label">موضوع</label>
                                <div style="padding: 0.75rem; background: var(--dark-700); border-radius: 0.5rem; border: 1px solid var(--dark-600); word-wrap: break-word; overflow-wrap: break-word;">
                                    ${bugData.bug.subject}
                                </div>
                            </div>
                            
                            <div class="form-group">
                                <label class="form-label">توضیحات</label>
                                <div style="padding: 0.75rem; background: var(--dark-700); border-radius: 0.5rem; border: 1px solid var(--dark-600); white-space: pre-wrap; word-wrap: break-word; overflow-wrap: break-word; line-height: 1.6; max-height: none; min-height: 100px;">
                                    ${bugData.bug.description}
                                </div>
                            </div>
                            
                            <div class="form-group">
                                <label class="form-label">گزارش‌دهنده</label>
                                <div style="padding: 0.75rem; background: var(--dark-700); border-radius: 0.5rem; border: 1px solid var(--dark-600); word-wrap: break-word; overflow-wrap: break-word;">
                                    ${bugData.bug.name} (${bugData.bug.email})
                                </div>
                            </div>
                            
                            <div class="form-group">
                                <label class="form-label">شناسه گزارش</label>
                                <div style="padding: 0.75rem; background: var(--dark-700); border-radius: 0.5rem; border: 1px solid var(--dark-600);">
                                    #${bugData.bug.id}
                                </div>
                            </div>
                        `;
                        document.getElementById('viewBugContent').innerHTML = content;
                    } else {
                        document.getElementById('viewBugContent').innerHTML = '<div style="text-align: center; padding: 2rem; color: var(--dark-300);">خطا در بارگذاری اطلاعات</div>';
                    }
                } catch (e) {
                    document.getElementById('viewBugContent').innerHTML = '<div style="text-align: center; padding: 2rem; color: var(--dark-300);">خطا در پردازش اطلاعات</div>';
                }
            })
            .catch(error => {
                document.getElementById('viewBugContent').innerHTML = '<div style="text-align: center; padding: 2rem; color: var(--dark-300);">خطا در ارتباط با سرور</div>';
            });
        }
        
        // Legacy view bug modal (for backward compatibility)
        function viewBug(bugId, subject, description, name, email) {
            // This function is kept for backward compatibility
            const content = `
                <div class="form-group">
                    <label class="form-label">موضوع</label>
                    <div style="padding: 0.75rem; background: var(--dark-700); border-radius: 0.5rem; border: 1px solid var(--dark-600); word-wrap: break-word; overflow-wrap: break-word;">
                        ${subject}
                    </div>
                </div>
                
                <div class="form-group">
                    <label class="form-label">توضیحات</label>
                    <div style="padding: 0.75rem; background: var(--dark-700); border-radius: 0.5rem; border: 1px solid var(--dark-600); white-space: pre-wrap; word-wrap: break-word; overflow-wrap: break-word; line-height: 1.6; max-height: none; min-height: 100px;">
                        ${description}
                    </div>
                </div>
                
                <div class="form-group">
                    <label class="form-label">گزارش‌دهنده</label>
                    <div style="padding: 0.75rem; background: var(--dark-700); border-radius: 0.5rem; border: 1px solid var(--dark-600); word-wrap: break-word; overflow-wrap: break-word;">
                        ${name} (${email})
                    </div>
                </div>
                
                <div class="form-group">
                    <label class="form-label">شناسه گزارش</label>
                    <div style="padding: 0.75rem; background: var(--dark-700); border-radius: 0.5rem; border: 1px solid var(--dark-600);">
                        #${bugId}
                    </div>
                </div>
            `;
            
            document.getElementById('viewBugContent').innerHTML = content;
            document.getElementById('viewModal').style.display = 'block';
        }
        
        function closeViewModal() {
            document.getElementById('viewModal').style.display = 'none';
        }
        
        // Bulk update modal
        function showBulkUpdateModal() {
            const checkboxes = document.querySelectorAll('.bug-checkbox:checked');
            
            if (checkboxes.length === 0) {
                alert('لطفاً حداقل یک گزارش را انتخاب کنید');
                return;
            }
            
            // Clear previous form data
            document.getElementById('bulkStatus').value = '';
            document.getElementById('bulkPriority').value = '';
            
            // Add selected bug IDs to the form
            const selectedBugsContainer = document.getElementById('bulkUpdateSelectedBugs');
            selectedBugsContainer.innerHTML = '';
            
            checkboxes.forEach(checkbox => {
                const hiddenInput = document.createElement('input');
                hiddenInput.type = 'hidden';
                hiddenInput.name = 'selected_bugs[]';
                hiddenInput.value = checkbox.value;
                selectedBugsContainer.appendChild(hiddenInput);
            });
            
            // Show count of selected items
            document.getElementById('bulkUpdateCount').textContent = `${checkboxes.length} گزارش انتخاب شده است`;
            
            document.getElementById('bulkUpdateModal').style.display = 'block';
        }
        
        function closeBulkUpdateModal() {
            document.getElementById('bulkUpdateModal').style.display = 'none';
        }
        
        // Delete single bug
        function deleteBug(bugId) {
            if (confirm('آیا از حذف این گزارش باگ اطمینان دارید؟')) {
                const form = document.createElement('form');
                form.method = 'POST';
                form.innerHTML = `
                    <input type="hidden" name="action" value="delete">
                    <input type="hidden" name="bug_id" value="${bugId}">
                `;
                document.body.appendChild(form);
                form.submit();
            }
        }
        
        // Close modals when clicking outside
        window.onclick = function(event) {
            const editModal = document.getElementById('editModal');
            const viewModal = document.getElementById('viewModal');
            const bulkUpdateModal = document.getElementById('bulkUpdateModal');
            
            if (event.target === editModal) {
                closeEditModal();
            }
            if (event.target === viewModal) {
                closeViewModal();
            }
            if (event.target === bulkUpdateModal) {
                closeBulkUpdateModal();
            }
        }
        
        // Close modals when pressing Escape key
        document.addEventListener('keydown', function(event) {
            if (event.key === 'Escape') {
                closeEditModal();
                closeViewModal();
                closeBulkUpdateModal();
            }
        });
        
        // Form validation
        document.getElementById('bulkUpdateForm').addEventListener('submit', function(e) {
            const status = document.getElementById('bulkStatus').value;
            const priority = document.getElementById('bulkPriority').value;
            
            if (!status && !priority) {
                e.preventDefault();
                alert('لطفاً حداقل یک فیلد را انتخاب کنید');
                return false;
            }
            
            // Show confirmation
            const checkboxes = document.querySelectorAll('.bug-checkbox:checked');
            if (!confirm(`آیا از بروزرسانی ${checkboxes.length} گزارش انتخاب شده اطمینان دارید؟`)) {
                e.preventDefault();
                return false;
            }
        });
    </script>
</body>
</html>
