<?php
// Set proper charset headers
header('Content-Type: text/html; charset=utf-8');
mb_internal_encoding('UTF-8');
mb_http_output('UTF-8');

// Start session
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once 'config/database.php';
require_once 'includes/functions.php';

// Check authentication
if (!isLoggedIn()) {
    logSecurityEvent('unauthorized_access_attempt', 'Unauthorized access attempt to dashboard.php from IP: ' . getClientIP(), 'high');
    header('Location: 403.php');
    exit;
}
checkSessionTimeout();

$message = '';
$error = '';

try {
    $db = new Database();
    $conn = $db->getConnection();
    
    // Get statistics
    $stats = [];
    
    // Total users
    $userQuery = "SELECT COUNT(*) as count FROM admin_users WHERE is_active = 1";
    $userStmt = $conn->query($userQuery);
    $stats['users'] = $userStmt->fetch()['count'];
    
    // Total bug reports
    $bugQuery = "SELECT COUNT(*) as count FROM bug_reports";
    $bugStmt = $conn->query($bugQuery);
    $stats['bugs'] = $bugStmt->fetch()['count'];
    
    // New bug reports
    $newBugQuery = "SELECT COUNT(*) as count FROM bug_reports WHERE status = 'new'";
    $newBugStmt = $conn->query($newBugQuery);
    $stats['new_bugs'] = $newBugStmt->fetch()['count'];
    
    // Total feature requests
    $featureQuery = "SELECT COUNT(*) as count FROM feature_requests";
    $featureStmt = $conn->query($featureQuery);
    $stats['features'] = $featureStmt->fetch()['count'];
    
    // New feature requests
    $newFeatureQuery = "SELECT COUNT(*) as count FROM feature_requests WHERE status = 'new'";
    $newFeatureStmt = $conn->query($newFeatureQuery);
    $stats['new_features'] = $newFeatureStmt->fetch()['count'];
    
    // Security events today
    $securityQuery = "SELECT COUNT(*) as count FROM security_logs WHERE DATE(created_at) = CURDATE()";
    $securityStmt = $conn->query($securityQuery);
    $stats['security_events'] = $securityStmt->fetch()['count'];
    
    // Recent bug reports
    $recentBugsQuery = "SELECT * FROM bug_reports ORDER BY created_at DESC LIMIT 5";
    $recentBugsStmt = $conn->query($recentBugsQuery);
    $recentBugs = $recentBugsStmt->fetchAll();
    
    // Recent feature requests
    $recentFeaturesQuery = "SELECT * FROM feature_requests ORDER BY created_at DESC LIMIT 5";
    $recentFeaturesStmt = $conn->query($recentFeaturesQuery);
    $recentFeatures = $recentFeaturesStmt->fetchAll();
    
    // Recent security events
    $recentSecurityQuery = "SELECT * FROM security_logs ORDER BY created_at DESC LIMIT 10";
    $recentSecurityStmt = $conn->query($recentSecurityQuery);
    $recentSecurity = $recentSecurityStmt->fetchAll();
    
} catch (Exception $e) {
    error_log("Dashboard error: " . $e->getMessage());
    $error = 'خطا در بارگذاری اطلاعات';
}
?>
<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>داشبورد - پنل مدیریت گواهیتو</title>
    <link rel="stylesheet" href="assets/admin-style.css">
    <link rel="icon" type="image/png" href="assets/images/logo.png">
    
        <?php include 'includes/admin-styles.php'; ?>
</head>
<body>
    <?php include 'includes/header.php'; ?>
    <?php include 'includes/sidebar.php'; ?>
    
    <!-- Sidebar Overlay for Mobile -->
    <div class="sidebar-overlay" id="sidebarOverlay"></div>

    <!-- Main Content -->
    <div class="main-content" id="mainContent">
        <div class="page-header">
            <h1 class="page-title">📊 داشبورد مدیریت</h1>
            <p class="page-description">خوش آمدید! اینجا می‌توانید وضعیت سیستم را مشاهده کنید</p>
        </div>

        <?php if ($error): ?>
            <div class="alert alert-danger">
                <?php echo htmlspecialchars($error); ?>
            </div>
        <?php endif; ?>

        <?php if ($message): ?>
            <div class="alert alert-success">
                <?php echo htmlspecialchars($message); ?>
            </div>
        <?php endif; ?>

        <!-- Statistics Cards -->
        <div class="grid grid-cols-1 grid-cols-2 grid-cols-3 grid-cols-4" style="margin-bottom: 2rem;">
            <div class="stats-card">
                <div class="stats-number"><?php echo $stats['users'] ?? 0; ?></div>
                <div class="stats-label">کاربران فعال</div>
            </div>
            
            <div class="stats-card">
                <div class="stats-number"><?php echo $stats['bugs'] ?? 0; ?></div>
                <div class="stats-label">گزارش‌های باگ</div>
            </div>
            
            <div class="stats-card">
                <div class="stats-number"><?php echo $stats['new_bugs'] ?? 0; ?></div>
                <div class="stats-label">باگ‌های جدید</div>
            </div>
            
            <div class="stats-card">
                <div class="stats-number"><?php echo $stats['features'] ?? 0; ?></div>
                <div class="stats-label">درخواست ویژگی</div>
            </div>
            
            <div class="stats-card">
                <div class="stats-number"><?php echo $stats['new_features'] ?? 0; ?></div>
                <div class="stats-label">درخواست‌های جدید</div>
            </div>
            
            <div class="stats-card">
                <div class="stats-number"><?php echo $stats['security_events'] ?? 0; ?></div>
                <div class="stats-label">رویدادهای امنیتی امروز</div>
            </div>
        </div>

        <!-- Recent Activity -->
        <div class="grid grid-cols-1 grid-cols-2" style="gap: 2rem;">
            <!-- Recent Bug Reports -->
            <div class="card">
                <div class="card-header">
                    <h2 class="card-title">🐛 آخرین گزارش‌های باگ</h2>
                    <a href="bug-reports.php" class="btn btn-primary btn-sm">مشاهده همه</a>
                </div>
                
                <?php if (!empty($recentBugs)): ?>
                    <div class="table-container">
                        <table class="table">
                            <thead>
                                <tr>
                                    <th>موضوع</th>
                                    <th>اولویت</th>
                                    <th>وضعیت</th>
                                    <th>تاریخ</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($recentBugs as $bug): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($bug['subject']); ?></td>
                                        <td><?php echo getPriorityBadge($bug['priority']); ?></td>
                                        <td><?php echo getStatusBadge($bug['status']); ?></td>
                                        <td><?php echo formatDate($bug['created_at']); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php else: ?>
                    <p style="color: var(--dark-400); text-align: center; padding: 2rem;">هیچ گزارش باگی یافت نشد</p>
                <?php endif; ?>
            </div>

            <!-- Recent Feature Requests -->
            <div class="card">
                <div class="card-header">
                    <h2 class="card-title">✨ آخرین درخواست‌های ویژگی</h2>
                    <a href="feature-requests.php" class="btn btn-primary btn-sm">مشاهده همه</a>
                </div>
                
                <?php if (!empty($recentFeatures)): ?>
                    <div class="table-container">
                        <table class="table">
                            <thead>
                                <tr>
                                    <th>ویژگی</th>
                                    <th>اولویت</th>
                                    <th>وضعیت</th>
                                    <th>تاریخ</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($recentFeatures as $feature): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($feature['feature']); ?></td>
                                        <td><?php echo getPriorityBadge($feature['priority']); ?></td>
                                        <td><?php echo getStatusBadge($feature['status']); ?></td>
                                        <td><?php echo formatDate($feature['created_at']); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php else: ?>
                    <p style="color: var(--dark-400); text-align: center; padding: 2rem;">هیچ درخواست ویژگی‌ای یافت نشد</p>
                <?php endif; ?>
            </div>
        </div>

        <!-- Security Events -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">🔒 آخرین رویدادهای امنیتی</h2>
                <a href="security.php" class="btn btn-primary btn-sm">مشاهده همه</a>
            </div>
            
            <?php if (!empty($recentSecurity)): ?>
                <div class="table-container">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>نوع رویداد</th>
                                <th>آدرس IP</th>
                                <th>سطح اهمیت</th>
                                <th>جزئیات</th>
                                <th>تاریخ</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($recentSecurity as $event): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($event['event_type']); ?></td>
                                    <td><code><?php echo htmlspecialchars($event['ip_address']); ?></code></td>
                                    <td>
                                        <?php
                                         $severity = $event['severity'];
                                         $severityClass = 'badge-info';
                                         if ($severity === 'medium') {
                                             $severityClass = 'badge-warning';
                                         } elseif ($severity === 'high' || $severity === 'critical') {
                                             $severityClass = 'badge-danger';
                                         }
                                         ?>
                                        <span class="badge <?php echo $severityClass; ?>">
                                            <?php echo htmlspecialchars($event['severity']); ?>
                                        </span>
                                    </td>
                                    <td><?php echo htmlspecialchars($event['details']); ?></td>
                                    <td><?php echo formatDate($event['created_at']); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php else: ?>
                <p style="color: var(--dark-400); text-align: center; padding: 2rem;">هیچ رویداد امنیتی‌ای یافت نشد</p>
            <?php endif; ?>
        </div>

        <!-- Quick Actions -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">⚡ اقدامات سریع</h2>
            </div>
            
            <div class="grid grid-cols-1 grid-cols-2 grid-cols-3" style="gap: 1rem;">
                <a href="users.php" class="btn btn-primary" style="justify-content: center;">
                    👥 مدیریت کاربران
                </a>
                <a href="bug-reports.php" class="btn btn-warning" style="justify-content: center;">
                    🐛 مشاهده باگ‌ها
                </a>
                <a href="feature-requests.php" class="btn btn-secondary" style="justify-content: center;">
                    ✨ درخواست‌های ویژگی
                </a>
                <a href="security.php" class="btn btn-danger" style="justify-content: center;">
                    🔒 امنیت سیستم
                </a>
                <a href="ip-management.php" class="btn btn-info" style="justify-content: center;">
                    🌐 مدیریت IP
                </a>
                <a href="settings.php" class="btn btn-success" style="justify-content: center;">
                    ⚙️ تنظیمات
                </a>
            </div>
        </div>
    </div>

    <script>
        // Auto-refresh dashboard every 30 seconds
        setTimeout(function() {
            location.reload();
        }, 30000);
        

    </script>
</body>
</html>
