<?php
header('Content-Type: text/html; charset=utf-8');
mb_internal_encoding('UTF-8');
mb_http_output('UTF-8');

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once 'config/database.php';
require_once 'includes/functions.php';

// Check authentication and admin privileges
if (!isLoggedIn()) {
    logSecurityEvent('unauthorized_access_attempt', 'Unauthorized access attempt to download-backup.php from IP: ' . getClientIP(), 'high');
    header('Location: 403.php');
    exit;
}
requireAdmin(); // Only admins can download backups
checkSessionTimeout();

// Check if IP is blacklisted
if (isIPBlacklisted(getClientIP())) {
    logSecurityEvent('blocked_download_attempt', 'IP blacklisted: ' . getClientIP(), 'high');
    header('Location: 403.php');
    exit;
}

// Rate limiting for download endpoint
if (!checkRateLimit('backup_download', 3)) { // Max 3 downloads per hour
    logSecurityEvent('rate_limit_exceeded', 'Backup download rate limit exceeded for IP: ' . getClientIP(), 'medium');
    header('Location: backups.php?error=' . urlencode('تعداد درخواست‌های شما بیش از حد مجاز است. لطفاً بعداً تلاش کنید'));
    exit;
}

// CSRF protection
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!isset($_POST['csrf_token']) || $_POST['csrf_token'] !== $_SESSION['csrf_token']) {
        logSecurityEvent('csrf_attempt', 'CSRF token mismatch in backup download', 'high');
        header('Location: 403.php');
        exit;
    }
} else {
    // For GET requests, validate referer
    if (!isset($_SERVER['HTTP_REFERER']) || !strpos($_SERVER['HTTP_REFERER'], $_SERVER['HTTP_HOST'])) {
        logSecurityEvent('invalid_referer', 'Invalid referer in backup download request', 'medium');
        header('Location: backups.php?error=' . urlencode('درخواست نامعتبر'));
        exit;
    }
}

$backupName = $_GET['backup'] ?? '';

if (empty($backupName)) {
    logSecurityEvent('empty_backup_name', 'Empty backup name in download request', 'low');
    header('Location: backups.php?error=' . urlencode('نام بکاپ مشخص نشده است'));
    exit;
}

// Enhanced backup name validation (prevent directory traversal and ensure alphanumeric + safe chars)
if (!preg_match('/^[a-zA-Z0-9._-]+$/', $backupName) || strlen($backupName) > 100) {
    logSecurityEvent('invalid_backup_name', 'Invalid backup name attempted: ' . $backupName, 'medium');
    header('Location: backups.php?error=' . urlencode('نام بکاپ نامعتبر است'));
    exit;
}

// Additional security: Check for suspicious patterns
if (preg_match('/(\.\.|php|\.|exec|system|shell)/i', $backupName)) {
    logSecurityEvent('suspicious_backup_name', 'Suspicious backup name pattern: ' . $backupName, 'high');
    header('Location: backups.php?error=' . urlencode('نام بکاپ نامعتبر است'));
    exit;
}

try {
    // Check if backup exists in database
    $db = new Database();
    $conn = $db->getConnection();
    
    $query = "SELECT * FROM backups WHERE backup_name = ? AND created_at IS NOT NULL";
    $stmt = $conn->prepare($query);
    $stmt->execute([$backupName]);
    $backup = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$backup) {
        logSecurityEvent('backup_not_found', 'Backup not found in database: ' . $backupName, 'low');
        header('Location: backups.php?error=' . urlencode('بکاپ یافت نشد'));
        exit;
    }
    
    // Additional check: Verify backup file actually exists on disk
    $currentDir = __DIR__;
    $projectRoot = dirname($currentDir);
    $backupPath = $projectRoot . '/backups/' . $backupName;
    
    if (!is_dir($backupPath) || !is_readable($backupPath)) {
        logSecurityEvent('backup_file_missing', 'Backup directory missing or not readable: ' . $backupPath, 'medium');
        header('Location: backups.php?error=' . urlencode('فایل بکاپ در دسترس نیست'));
        exit;
    }
    
    // Create zip file
    $result = downloadBackup($backupName);
    
    if (!$result['success']) {
        logSecurityEvent('backup_creation_failed', 'Failed to create backup zip: ' . $result['error'], 'medium');
        header('Location: backups.php?error=' . urlencode($result['error']));
        exit;
    }
    
    $zipPath = $result['zip_path'];
    $zipName = $result['zip_name'];
    
    if (!file_exists($zipPath) || !is_readable($zipPath)) {
        logSecurityEvent('zip_file_missing', 'ZIP file missing or not readable: ' . $zipPath, 'medium');
        header('Location: backups.php?error=' . urlencode('فایل ZIP ایجاد نشد'));
        exit;
    }
    
    // Security check: Verify zip file is actually a zip file
    $isValidZip = false;
    
    // Method 1: Check file extension
    if (pathinfo($zipPath, PATHINFO_EXTENSION) === 'zip') {
        // Method 2: Check file header (ZIP files start with PK\x03\x04)
        $handle = fopen($zipPath, 'rb');
        if ($handle) {
            $header = fread($handle, 4);
            fclose($handle);
            if ($header === 'PK' . chr(3) . chr(4)) {
                $isValidZip = true;
            }
        }
    }
    
    // Alternative method: Use fileinfo if available
    if (!$isValidZip && function_exists('finfo_open')) {
        $finfo = finfo_open(FILEINFO_MIME_TYPE);
        $mimeType = finfo_file($finfo, $zipPath);
        finfo_close($finfo);
        
        if ($mimeType === 'application/zip' || $mimeType === 'application/x-zip-compressed') {
            $isValidZip = true;
        }
    }
    
    if (!$isValidZip) {
        logSecurityEvent('invalid_zip_file', 'Invalid ZIP file format detected', 'high');
        unlink($zipPath); // Clean up
        header('Location: backups.php?error=' . urlencode('فایل ZIP نامعتبر است'));
        exit;
    }
    
    // Log successful download
    logSecurityEvent('backup_downloaded', "Backup downloaded successfully: {$backupName} by user: " . ($_SESSION['username'] ?? 'unknown'), 'low');
    
    // Set headers for file download
    header('Content-Type: application/zip');
    header('Content-Disposition: attachment; filename="' . $zipName . '"');
    header('Content-Length: ' . filesize($zipPath));
    header('Cache-Control: no-cache, must-revalidate');
    header('Expires: 0');
    header('X-Content-Type-Options: nosniff');
    header('X-Frame-Options: DENY');
    header('X-XSS-Protection: 1; mode=block');
    
    // Clear output buffer
    if (ob_get_level()) {
        ob_end_clean();
    }
    
    // Read and output file
    readfile($zipPath);
    
    // Delete temporary zip file
    unlink($zipPath);
    
    exit;
    
} catch (Exception $e) {
    error_log("Download backup error: " . $e->getMessage());
    logSecurityEvent('backup_download_error', 'Exception in backup download: ' . $e->getMessage(), 'medium');
    header('Location: backups.php?error=' . urlencode('خطا در دانلود بکاپ'));
    exit;
}
?>
