<?php
// شروع session اگر شروع نشده باشد
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// توابع امنیتی
function sanitizeInput($input) {
    if (is_array($input)) {
        return array_map('sanitizeInput', $input);
    }
    return htmlspecialchars(trim($input), ENT_QUOTES, 'UTF-8');
}

function validateEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL);
}

function getClientIP() {
    $ipKeys = ['HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_FORWARDED', 'HTTP_X_CLUSTER_CLIENT_IP', 'HTTP_FORWARDED_FOR', 'HTTP_FORWARDED', 'REMOTE_ADDR'];
    
    foreach ($ipKeys as $key) {
        if (array_key_exists($key, $_SERVER) === true) {
            foreach (explode(',', $_SERVER[$key]) as $ip) {
                $ip = trim($ip);
                if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) !== false) {
                    return $ip;
                }
            }
        }
    }
    
    return $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
}

function getUserAgent() {
    return $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown';
}

function detectXSS($input) {
    // Check if XSS protection is enabled
    if (getSystemSetting('enable_xss_protection') !== '1') {
        return false;
    }
    
    $xssPatterns = [
        '/<script[^>]*>.*?<\/script>/is',
        '/javascript:/i',
        '/on\w+\s*=/i',
        '/<iframe[^>]*>/i',
        '/<object[^>]*>/i',
        '/<embed[^>]*>/i',
        '/<form[^>]*>/i',
        '/<input[^>]*>/i',
        '/<textarea[^>]*>/i',
        '/<select[^>]*>/i',
        '/<button[^>]*>/i',
        '/<link[^>]*>/i',
        '/<meta[^>]*>/i',
        '/<style[^>]*>.*?<\/style>/is'
    ];
    
    foreach ($xssPatterns as $pattern) {
        if (preg_match($pattern, $input)) {
            return true;
        }
    }
    
    return false;
}

function detectSQLInjection($input) {
    // Check if SQL injection protection is enabled
    if (getSystemSetting('enable_sql_injection_protection') !== '1') {
        return false;
    }
    
    $sqlPatterns = [
        '/\b(union|select|insert|update|delete|drop|create|alter|exec|execute)\b/i',
        '/[\'";]/',
        '/--/',
        '/\/\*.*\*\//',
        '/xp_cmdshell/i',
        '/sp_executesql/i'
    ];
    
    foreach ($sqlPatterns as $pattern) {
        if (preg_match($pattern, $input)) {
            return true;
        }
    }
    
    return false;
}

// توابع احراز هویت
function isLoggedIn() {
    return isset($_SESSION['admin_id']) && isset($_SESSION['admin_username']);
}

function requireLogin() {
    if (!isLoggedIn()) {
        logSecurityEvent('unauthorized_access_attempt', 'Unauthorized access attempt from IP: ' . getClientIP() . ' to: ' . $_SERVER['REQUEST_URI'], 'high');
        header('Location: 403.php');
        exit();
    }
}

function requireAdmin() {
    requireLogin();
    if ($_SESSION['admin_role'] !== 'admin') {
        header('Location: 403.php');
        exit();
    }
}

function checkSessionTimeout() {
    $timeoutMinutes = (int)getSystemSetting('session_timeout_minutes') ?: 30;
    $timeout = $timeoutMinutes * 60; // Convert to seconds
    if (isset($_SESSION['last_activity']) && (time() - $_SESSION['last_activity'] > $timeout)) {
        logSecurityEvent('session_timeout', 'Session timeout for user: ' . ($_SESSION['admin_username'] ?? 'unknown') . ' from IP: ' . getClientIP(), 'medium');
        session_unset();
        session_destroy();
        header('Location: 403.php');
        exit();
    }
    $_SESSION['last_activity'] = time();
}

function logSecurityEvent($eventType, $details = '', $severity = 'medium') {
    try {
        $db = new Database();
        $conn = $db->getConnection();
        
        $query = "INSERT INTO security_logs (event_type, ip_address, user_agent, details, severity) VALUES (?, ?, ?, ?, ?)";
        $stmt = $conn->prepare($query);
        $stmt->execute([
            $eventType,
            getClientIP(),
            getUserAgent(),
            $details,
            $severity
        ]);
    } catch (Exception $e) {
        error_log("Error logging security event: " . $e->getMessage());
    }
}

function checkRateLimit($endpoint, $limit = null) {
    if ($limit === null) {
        $limit = (int)getSystemSetting('rate_limit_per_hour') ?: 5;
    }
    $ip = getClientIP();
    $timeWindow = date('Y-m-d H:i:00');
    
    try {
        $db = new Database();
        $conn = $db->getConnection();
        
        // Clean old records
        $cleanQuery = "DELETE FROM rate_limit_logs WHERE created_at < DATE_SUB(NOW(), INTERVAL 1 HOUR)";
        $conn->exec($cleanQuery);
        
        // Count recent requests
        $countQuery = "SELECT COUNT(*) as count FROM rate_limit_logs WHERE ip_address = ? AND endpoint = ? AND created_at >= ?";
        $countStmt = $conn->prepare($countQuery);
        $countStmt->execute([$ip, $endpoint, $timeWindow]);
        $result = $countStmt->fetch();
        
        if ($result['count'] >= $limit) {
            return false;
        }
        
        // Log this request
        $logQuery = "INSERT INTO rate_limit_logs (ip_address, endpoint) VALUES (?, ?)";
        $logStmt = $conn->prepare($logQuery);
        $logStmt->execute([$ip, $endpoint]);
        
        return true;
    } catch (Exception $e) {
        error_log("Rate limit check error: " . $e->getMessage());
        return true; // Allow if error
    }
}

function isIPBlacklisted($ip) {
    try {
        $db = new Database();
        $conn = $db->getConnection();
        
        $query = "SELECT id FROM ip_blacklist WHERE ip_address = ? AND (expires_at IS NULL OR expires_at > NOW())";
        $stmt = $conn->prepare($query);
        $stmt->execute([$ip]);
        
        return $stmt->fetch() !== false;
    } catch (Exception $e) {
        error_log("IP blacklist check error: " . $e->getMessage());
        return false;
    }
}

function isIPWhitelisted($ip) {
    try {
        $db = new Database();
        $conn = $db->getConnection();
        
        $query = "SELECT id FROM ip_whitelist WHERE ip_address = ? AND is_active = 1";
        $stmt = $conn->prepare($query);
        $stmt->execute([$ip]);
        
        return $stmt->fetch() !== false;
    } catch (Exception $e) {
        error_log("IP whitelist check error: " . $e->getMessage());
        return false;
    }
}

function getSystemSetting($key, $default = null) {
    try {
        $db = new Database();
        $conn = $db->getConnection();
        
        $query = "SELECT setting_value FROM system_settings WHERE setting_key = ?";
        $stmt = $conn->prepare($query);
        $stmt->execute([$key]);
        $result = $stmt->fetch();
        
        return $result ? $result['setting_value'] : $default;
    } catch (Exception $e) {
        error_log("System setting error: " . $e->getMessage());
        return $default;
    }
}

function setSystemSetting($key, $value) {
    try {
        $db = new Database();
        $conn = $db->getConnection();
        
        $query = "INSERT INTO system_settings (setting_key, setting_value) VALUES (?, ?) 
                  ON DUPLICATE KEY UPDATE setting_value = ?, updated_at = NOW()";
        $stmt = $conn->prepare($query);
        $stmt->execute([$key, $value, $value]);
        
        return true;
    } catch (Exception $e) {
        error_log("System setting update error: " . $e->getMessage());
        return false;
    }
}

function formatDate($date) {
    if (!$date) return 'نامشخص';
    
    $timestamp = strtotime($date);
    if ($timestamp === false) return 'نامشخص';
    
    // Check if date is today
    $today = date('Y-m-d');
    $dateOnly = date('Y-m-d', $timestamp);
    
    if ($dateOnly === $today) {
        return 'امروز ' . date('H:i', $timestamp);
    }
    
    // Check if date is yesterday
    $yesterday = date('Y-m-d', strtotime('-1 day'));
    if ($dateOnly === $yesterday) {
        return 'دیروز ' . date('H:i', $timestamp);
    }
    
    // For other dates, show full date
    return date('Y/m/d H:i', $timestamp);
}

function getStatusBadge($status) {
    $badges = [
        'new' => '<span class="px-2 py-1 text-xs font-medium bg-blue-100 text-blue-800 rounded-full">جدید</span>',
        'in_progress' => '<span class="px-2 py-1 text-xs font-medium bg-yellow-100 text-yellow-800 rounded-full">در حال بررسی</span>',
        'resolved' => '<span class="px-2 py-1 text-xs font-medium bg-green-100 text-green-800 rounded-full">حل شده</span>',
        'closed' => '<span class="px-2 py-1 text-xs font-medium bg-gray-100 text-gray-800 rounded-full">بسته</span>',
        'approved' => '<span class="px-2 py-1 text-xs font-medium bg-green-100 text-green-800 rounded-full">تایید شده</span>',
        'rejected' => '<span class="px-2 py-1 text-xs font-medium bg-red-100 text-red-800 rounded-full">رد شده</span>',
        'completed' => '<span class="px-2 py-1 text-xs font-medium bg-green-100 text-green-800 rounded-full">تکمیل شده</span>'
    ];
    
    return $badges[$status] ?? '<span class="px-2 py-1 text-xs font-medium bg-gray-100 text-gray-800 rounded-full">نامشخص</span>';
}

function getPriorityBadge($priority) {
    $badges = [
        'low' => '<span class="px-2 py-1 text-xs font-medium bg-gray-100 text-gray-800 rounded-full">کم</span>',
        'medium' => '<span class="px-2 py-1 text-xs font-medium bg-yellow-100 text-yellow-800 rounded-full">متوسط</span>',
        'high' => '<span class="px-2 py-1 text-xs font-medium bg-orange-100 text-orange-800 rounded-full">زیاد</span>',
        'critical' => '<span class="px-2 py-1 text-xs font-medium bg-red-100 text-red-800 rounded-full">بحرانی</span>'
    ];
    
    return $badges[$priority] ?? '<span class="px-2 py-1 text-xs font-medium bg-gray-100 text-gray-800 rounded-full">نامشخص</span>';
}

function getFailedLoginAttempts($ip) {
    try {
        $db = new Database();
        $conn = $db->getConnection();
        
        // Get max login attempts from settings
        $maxAttempts = (int)getSystemSetting('max_login_attempts') ?: 5;
        
        // Count failed login attempts in the last hour
        $query = "SELECT COUNT(*) as count FROM security_logs 
                  WHERE ip_address = ? 
                  AND event_type = 'login_failed_hidden' 
                  AND created_at >= DATE_SUB(NOW(), INTERVAL 1 HOUR)";
        $stmt = $conn->prepare($query);
        $stmt->execute([$ip]);
        $result = $stmt->fetch();
        
        return (int)$result['count'];
    } catch (Exception $e) {
        error_log("Failed login attempts check error: " . $e->getMessage());
        return 0;
    }
}

// IP Restriction Functions
function isAdminIPRestrictionEnabled() {
    try {
        $db = new Database();
        $conn = $db->getConnection();
        
        $query = "SELECT COUNT(*) as count FROM admin_ip_restriction WHERE is_active = 1";
        $stmt = $conn->prepare($query);
        $stmt->execute();
        $result = $stmt->fetch();
        
        return (int)$result['count'] > 0;
    } catch (Exception $e) {
        error_log("Admin IP restriction check error: " . $e->getMessage());
        return false;
    }
}

function isAllowedAdminIP($ip) {
    try {
        $db = new Database();
        $conn = $db->getConnection();
        
        // First check admin_ip_restriction table
        $query = "SELECT * FROM admin_ip_restriction WHERE allowed_ip = ? AND is_active = 1";
        $stmt = $conn->prepare($query);
        $stmt->execute([$ip]);
        
        if ($stmt->fetch() !== false) {
            return true;
        }
        
        // Then check ip_whitelist table
        $whitelistQuery = "SELECT * FROM ip_whitelist WHERE ip_address = ?";
        $whitelistStmt = $conn->prepare($whitelistQuery);
        $whitelistStmt->execute([$ip]);
        
        if ($whitelistStmt->fetch() !== false) {
            return true;
        }
        
        return false;
    } catch (Exception $e) {
        error_log("Allowed admin IP check error: " . $e->getMessage());
        return false;
    }
}

function setAdminIPRestriction($ip, $adminEmail) {
    try {
        $db = new Database();
        $conn = $db->getConnection();
        
        // Clear existing restrictions
        $clearQuery = "UPDATE admin_ip_restriction SET is_active = 0";
        $conn->exec($clearQuery);
        
        // Add new restriction
        $insertQuery = "INSERT INTO admin_ip_restriction (allowed_ip, admin_email, is_active) VALUES (?, ?, 1)";
        $stmt = $conn->prepare($insertQuery);
        $stmt->execute([$ip, $adminEmail]);
        
        // Log the action
        logSecurityEvent('admin_ip_restriction_set', 'IP: ' . $ip . ', Email: ' . $adminEmail, 'high', json_encode([
            'ip' => $ip,
            'admin_email' => $adminEmail
        ]));
        
        return true;
    } catch (Exception $e) {
        error_log("Set admin IP restriction error: " . $e->getMessage());
        return false;
    }
}

function removeAdminIPRestriction() {
    try {
        $db = new Database();
        $conn = $db->getConnection();
        
        $query = "UPDATE admin_ip_restriction SET is_active = 0";
        $conn->exec($query);
        
        // Log the action
        logSecurityEvent('admin_ip_restriction_removed', 'All IP restrictions removed', 'high');
        
        return true;
    } catch (Exception $e) {
        error_log("Remove admin IP restriction error: " . $e->getMessage());
        return false;
    }
}

function getCurrentAdminIPRestriction() {
    try {
        $db = new Database();
        $conn = $db->getConnection();
        
        $query = "SELECT * FROM admin_ip_restriction WHERE is_active = 1 ORDER BY created_at DESC LIMIT 1";
        $stmt = $conn->prepare($query);
        $stmt->execute();
        
        return $stmt->fetch();
    } catch (Exception $e) {
        error_log("Get current admin IP restriction error: " . $e->getMessage());
        return null;
    }
}

function generateEmergencyToken($adminEmail) {
    try {
        $db = new Database();
        $conn = $db->getConnection();
        
        // Generate unique token
        $token = bin2hex(random_bytes(32));
        $expiresAt = date('Y-m-d H:i:s', strtotime('+1 hour'));
        
        $insertQuery = "INSERT INTO admin_ip_tokens (token, admin_email, expires_at) VALUES (?, ?, ?)";
        $stmt = $conn->prepare($insertQuery);
        $stmt->execute([$token, $adminEmail, $expiresAt]);
        
        return $token;
    } catch (Exception $e) {
        error_log("Generate emergency token error: " . $e->getMessage());
        return null;
    }
}

function validateEmergencyToken($token) {
    try {
        $db = new Database();
        $conn = $db->getConnection();
        
        $query = "SELECT * FROM admin_ip_tokens WHERE token = ? AND is_used = 0 AND expires_at > NOW()";
        $stmt = $conn->prepare($query);
        $stmt->execute([$token]);
        $result = $stmt->fetch();
        
        if ($result) {
            // Mark token as used
            $updateQuery = "UPDATE admin_ip_tokens SET is_used = 1 WHERE token = ?";
            $updateStmt = $conn->prepare($updateQuery);
            $updateStmt->execute([$token]);
            
            // Remove IP restrictions
            removeAdminIPRestriction();
            
            return true;
        }
        
        return false;
    } catch (Exception $e) {
        error_log("Validate emergency token error: " . $e->getMessage());
        return false;
    }
}

function sendEmergencyEmail($adminEmail, $token) {
    try {
        require_once __DIR__ . '/../config/email-config.php';
        require_once __DIR__ . '/../vendor/autoload.php';
        
        $mail = new PHPMailer\PHPMailer\PHPMailer(true);
        
        // Server settings
        $mail->isSMTP();
        $mail->Host = SMTP_HOST;
        $mail->SMTPAuth = true;
        $mail->Username = SMTP_USERNAME;
        $mail->Password = SMTP_PASSWORD;
        $mail->SMTPSecure = SMTP_SECURE;
        $mail->Port = SMTP_PORT;
        $mail->CharSet = 'UTF-8';
        
        // Recipients
        $mail->setFrom(SMTP_FROM_EMAIL, SMTP_FROM_NAME);
        $mail->addAddress($adminEmail);
        
        // Content
        $mail->isHTML(true);
        $mail->Subject = '🔓 دسترسی اضطراری به پنل ادمین - یادینو';
        
        $emergencyUrl = 'https://gavahito.ir/backend/Hua89fe754tr7gyu7o9werdhy79erwte0gw04etyg.php?token=' . $token;
        
        $mail->Body = "
        <div style='font-family: Tahoma, Arial, sans-serif; direction: rtl; text-align: right;'>
            <h2 style='color: #e74c3c;'>🔓 دسترسی اضطراری به پنل ادمین</h2>
            <p>سلام،</p>
            <p>یک درخواست دسترسی اضطراری برای پنل ادمین ثبت شده است.</p>
            <p><strong>توجه:</strong> این لینک فقط 1 ساعت معتبر است.</p>
            <div style='background: #f8f9fa; padding: 20px; border-radius: 10px; margin: 20px 0;'>
                <p><strong>لینک دسترسی اضطراری:</strong></p>
                <a href='{$emergencyUrl}' style='color: #3498db; text-decoration: none; font-weight: bold;'>
                    {$emergencyUrl}
                </a>
            </div>
            <p style='color: #e74c3c; font-weight: bold;'>⚠️ اگر شما این درخواست را نکرده‌اید، این ایمیل را نادیده بگیرید.</p>
            <hr>
            <p style='font-size: 12px; color: #7f8c8d;'>این ایمیل از سیستم امنیتی یادینو ارسال شده است.</p>
        </div>";
        
        $mail->send();
        return true;
    } catch (Exception $e) {
        error_log("Send emergency email error: " . $e->getMessage());
        return false;
    }
}

// Emergency Access Code Functions
function generateEmergencyAccessCode($adminEmail) {
    try {
        $db = new Database();
        $conn = $db->getConnection();
        
        // Generate 6-digit code
        $accessCode = sprintf('%06d', mt_rand(0, 999999));
        $expiresAt = date('Y-m-d H:i:s', strtotime('+10 minutes'));
        
        // Clear old codes for this email
        $clearQuery = "DELETE FROM admin_ip_tokens WHERE admin_email = ? AND token LIKE 'ACCESS_CODE_%'";
        $clearStmt = $conn->prepare($clearQuery);
        $clearStmt->execute([$adminEmail]);
        
        // Insert new code
        $insertQuery = "INSERT INTO admin_ip_tokens (token, admin_email, expires_at) VALUES (?, ?, ?)";
        $stmt = $conn->prepare($insertQuery);
        $stmt->execute(['ACCESS_CODE_' . $accessCode, $adminEmail, $expiresAt]);
        
        return $accessCode;
    } catch (Exception $e) {
        error_log("Generate emergency access code error: " . $e->getMessage());
        return null;
    }
}

function validateEmergencyAccessCode($accessCode, $adminEmail) {
    try {
        $db = new Database();
        $conn = $db->getConnection();
        
        $query = "SELECT * FROM admin_ip_tokens WHERE token = ? AND admin_email = ? AND is_used = 0 AND expires_at > NOW()";
        $stmt = $conn->prepare($query);
        $stmt->execute(['ACCESS_CODE_' . $accessCode, $adminEmail]);
        $result = $stmt->fetch();
        
        if ($result) {
            // Mark code as used
            $updateQuery = "UPDATE admin_ip_tokens SET is_used = 1 WHERE token = ?";
            $updateStmt = $conn->prepare($updateQuery);
            $updateStmt->execute(['ACCESS_CODE_' . $accessCode]);
            
            return true;
        }
        
        return false;
    } catch (Exception $e) {
        error_log("Validate emergency access code error: " . $e->getMessage());
        return false;
    }
}

function sendEmergencyAccessEmail($adminEmail, $accessCode) {
    try {
        require_once __DIR__ . '/../config/email-config.php';
        require_once __DIR__ . '/../vendor/autoload.php';
        
        $mail = new PHPMailer\PHPMailer\PHPMailer(true);
        
        // Server settings
        $mail->isSMTP();
        $mail->Host = SMTP_HOST;
        $mail->SMTPAuth = true;
        $mail->Username = SMTP_USERNAME;
        $mail->Password = SMTP_PASSWORD;
        $mail->SMTPSecure = SMTP_SECURE;
        $mail->Port = SMTP_PORT;
        $mail->CharSet = 'UTF-8';
        
        // Recipients
        $mail->setFrom(SMTP_FROM_EMAIL, SMTP_FROM_NAME);
        $mail->addAddress($adminEmail);
        
        // Content
        $mail->isHTML(true);
        $mail->Subject = '🔐 کد دسترسی اضطراری - پنل ادمین یادینو';
        
        $mail->Body = "
        <div style='font-family: Tahoma, Arial, sans-serif; direction: rtl; text-align: right;'>
            <h2 style='color: #e74c3c;'>🔐 کد دسترسی اضطراری</h2>
            <p>سلام،</p>
            <p>یک درخواست کد دسترسی اضطراری برای پنل ادمین ثبت شده است.</p>
            <p><strong>توجه:</strong> این کد فقط 10 دقیقه معتبر است.</p>
            <div style='background: #f8f9fa; padding: 20px; border-radius: 10px; margin: 20px 0; text-align: center;'>
                <p style='font-size: 24px; font-weight: bold; color: #e74c3c; margin: 0;'>
                    کد دسترسی: <span style='background: #e74c3c; color: white; padding: 10px 20px; border-radius: 8px;'>{$accessCode}</span>
                </p>
            </div>
            <p style='color: #e74c3c; font-weight: bold;'>⚠️ اگر شما این درخواست را نکرده‌اید، این ایمیل را نادیده بگیرید.</p>
            <hr>
            <p style='font-size: 12px; color: #7f8c8d;'>این ایمیل از سیستم امنیتی یادینو ارسال شده است.</p>
        </div>";
        
        $mail->send();
        return true;
    } catch (Exception $e) {
        error_log("Send emergency access email error: " . $e->getMessage());
        return false;
    }
}

// Backup Functions
function createBackup($type = 'full') {
    try {
        // Get the correct backup directory path
        $currentDir = __DIR__; // backend/includes
        $projectRoot = dirname(dirname($currentDir)); // Go up two levels to project root
        $backupDir = $projectRoot . '/backups/';
        
        if (!is_dir($backupDir)) {
            mkdir($backupDir, 0755, true);
        }
        
        $timestamp = date('Y-m-d_H-i-s');
        
        // Check if this is an automatic backup (called from cron)
        $isAutomatic = false;
        
        // Check if called from cron script
        $backtrace = debug_backtrace();
        foreach ($backtrace as $trace) {
            if (isset($trace['file']) && strpos($trace['file'], 'auto-backup.php') !== false) {
                $isAutomatic = true;
                break;
            }
        }
        
        $backupName = "backup_{$type}_{$timestamp}" . ($isAutomatic ? "_auto" : "");
        $backupPath = $backupDir . $backupName;
        

        
        // Create backup directory
        if (!is_dir($backupPath)) {
            mkdir($backupPath, 0755, true);
        }
        
        $backupInfo = [
            'type' => $type,
            'created_at' => date('Y-m-d H:i:s'),
            'files_count' => 0,
            'database_size' => 0,
            'total_size' => 0,
            'status' => 'in_progress'
        ];
        
        // Backup database
        $dbBackupPath = $backupPath . '/database.sql';
        if (backupDatabase($dbBackupPath)) {
            $backupInfo['database_size'] = filesize($dbBackupPath);
        }
        
        // Backup files
        $filesBackupPath = $backupPath . '/files/';
        $backupInfo['files_count'] = backupFiles($filesBackupPath);
        
        // Calculate total size
        $backupInfo['total_size'] = calculateDirectorySize($backupPath);
        $backupInfo['status'] = 'completed';
        
        // Save backup info to database
        saveBackupInfo($backupName, $backupInfo);
        
        // Create backup info file
        file_put_contents($backupPath . '/backup_info.json', json_encode($backupInfo, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
        
        logSecurityEvent('backup_created', "Backup created: {$backupName}", 'low');
        
        return [
            'success' => true,
            'backup_name' => $backupName,
            'info' => $backupInfo
        ];
        
    } catch (Exception $e) {
        error_log("Backup creation error: " . $e->getMessage());
        return [
            'success' => false,
            'error' => $e->getMessage()
        ];
    }
}

function backupDatabase($outputPath) {
    try {
        $db = new Database();
        $conn = $db->getConnection();
        
        // Check if exec is available
        if (!function_exists('exec')) {
            // Use PHP-based backup instead of mysqldump
            return backupDatabasePHP($conn, $outputPath);
        }
        
        // Get database configuration
        $host = $db->host;
        $dbname = $db->database;
        $username = $db->username;
        $password = $db->password;
        
        // Create mysqldump command
        $command = "mysqldump --host={$host} --user={$username} --password={$password} {$dbname} > {$outputPath}";
        
        // Execute command
        $output = [];
        $returnCode = 0;
        exec($command . " 2>&1", $output, $returnCode);
        
        if ($returnCode !== 0) {
            // Fallback to PHP-based backup if mysqldump fails
            return backupDatabasePHP($conn, $outputPath);
        }
        
        return true;
    } catch (Exception $e) {
        error_log("Database backup error: " . $e->getMessage());
        return false;
    }
}

function backupDatabasePHP($conn, $outputPath) {
    try {
        $backup = "-- Database Backup\n";
        $backup .= "-- Generated: " . date('Y-m-d H:i:s') . "\n";
        $backup .= "-- Host: " . $_SERVER['HTTP_HOST'] . "\n\n";
        
        // Get all tables
        $tables = [];
        $result = $conn->query("SHOW TABLES");
        while ($row = $result->fetch(PDO::FETCH_NUM)) {
            $tables[] = $row[0];
        }
        
        foreach ($tables as $table) {
            $backup .= "-- Table structure for table `{$table}`\n";
            $backup .= "DROP TABLE IF EXISTS `{$table}`;\n";
            
            // Get create table statement
            $result = $conn->query("SHOW CREATE TABLE `{$table}`");
            $row = $result->fetch(PDO::FETCH_NUM);
            $backup .= $row[1] . ";\n\n";
            
            // Get table data
            $result = $conn->query("SELECT * FROM `{$table}`");
            $rows = $result->fetchAll(PDO::FETCH_ASSOC);
            
            if (!empty($rows)) {
                $backup .= "-- Data for table `{$table}`\n";
                $backup .= "INSERT INTO `{$table}` VALUES\n";
                
                $values = [];
                foreach ($rows as $row) {
                    $rowValues = [];
                    foreach ($row as $value) {
                        if ($value === null) {
                            $rowValues[] = 'NULL';
                        } else {
                            $rowValues[] = "'" . addslashes($value) . "'";
                        }
                    }
                    $values[] = "(" . implode(', ', $rowValues) . ")";
                }
                
                $backup .= implode(",\n", $values) . ";\n\n";
            }
        }
        
        // Write backup to file
        file_put_contents($outputPath, $backup);
        
        return true;
    } catch (Exception $e) {
        error_log("PHP Database backup error: " . $e->getMessage());
        return false;
    }
}

function backupFiles($outputPath) {
    try {
        if (!is_dir($outputPath)) {
            mkdir($outputPath, 0755, true);
        }
        
        $filesCount = 0;
        // Get the correct project root path
        $currentDir = __DIR__; // backend/includes
        $rootPath = dirname(dirname($currentDir)); // Go up two levels to project root
        

        
        // Define directories to backup (entire public_html)
        $directories = [
            'src',
            'public',
            'backend',
            'index.html',
            'index.php',
            '.htaccess',
            'assets',
            'images',
            'theme',
            'package.json',
            'package-lock.json',
            'tailwind.config.js',
            'postcss.config.js',
            'tsconfig.json',
            'tsconfig.node.json',
            'vite.config.ts',
            'README.md',
            'SETTINGS_INTEGRATION.md',
            'IP_CHECK_INTEGRATION.md',
            'SYSTEM_STATUS.md',
            'BACKUP_SYSTEM.md'
        ];
        
        // Define files to exclude
        $excludePatterns = [
            '/node_modules/',
            '/.git/',
            '/backups/',
            '/vendor/',
            '/dist/',
            '/dist-deploy/',
            '/.env',
            '/.DS_Store',
            '/Thumbs.db',
            '/backend/backups/',
            '/backend/logs/',
            '/backend/temp/',
            '/backend/cache/'
        ];
        
        foreach ($directories as $dir) {
            $sourcePath = $rootPath . '/' . $dir;
            
            if (is_file($sourcePath)) {
                // Copy single file
                $destPath = $outputPath . '/' . $dir;
                $destDir = dirname($destPath);
                
                if (!is_dir($destDir)) {
                    mkdir($destDir, 0755, true);
                }
                
                if (copy($sourcePath, $destPath)) {
                    $filesCount++;
                }
            } elseif (is_dir($sourcePath)) {
                // Copy directory recursively
                $filesCount += copyDirectory($sourcePath, $outputPath . '/' . $dir, $excludePatterns);
            }
        }
        
        return $filesCount;
    } catch (Exception $e) {
        error_log("Files backup error: " . $e->getMessage());
        return 0;
    }
}

function copyDirectory($source, $destination, $excludePatterns = []) {
    $filesCount = 0;
    
    if (!is_dir($destination)) {
        mkdir($destination, 0755, true);
    }
    
    $iterator = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($source, RecursiveDirectoryIterator::SKIP_DOTS),
        RecursiveIteratorIterator::SELF_FIRST
    );
    
    foreach ($iterator as $item) {
        $sourcePath = $item->getRealPath();
        $relativePath = str_replace($source . '/', '', $sourcePath);
        $destPath = $destination . '/' . $relativePath;
        
        // Check if path should be excluded
        $shouldExclude = false;
        foreach ($excludePatterns as $pattern) {
            if (strpos($sourcePath, $pattern) !== false) {
                $shouldExclude = true;
                break;
            }
        }
        
        if ($shouldExclude) {
            continue;
        }
        
        if ($item->isDir()) {
            if (!is_dir($destPath)) {
                mkdir($destPath, 0755, true);
            }
        } else {
            $destDir = dirname($destPath);
            if (!is_dir($destDir)) {
                mkdir($destDir, 0755, true);
            }
            
            if (copy($sourcePath, $destPath)) {
                $filesCount++;
            }
        }
    }
    
    return $filesCount;
}

function calculateDirectorySize($path) {
    $size = 0;
    
    if (is_file($path)) {
        return filesize($path);
    }
    
    $iterator = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($path, RecursiveDirectoryIterator::SKIP_DOTS)
    );
    
    foreach ($iterator as $file) {
        if ($file->isFile()) {
            $size += $file->getSize();
        }
    }
    
    return $size;
}

function saveBackupInfo($backupName, $info) {
    try {
        $db = new Database();
        $conn = $db->getConnection();
        
        $query = "INSERT INTO backups (backup_name, backup_type, created_at, files_count, database_size, total_size, status) 
                  VALUES (?, ?, ?, ?, ?, ?, ?)";
        $stmt = $conn->prepare($query);
        $stmt->execute([
            $backupName,
            $info['type'],
            $info['created_at'],
            $info['files_count'],
            $info['database_size'],
            $info['total_size'],
            $info['status']
        ]);
        
        return true;
    } catch (Exception $e) {
        error_log("Save backup info error: " . $e->getMessage());
        return false;
    }
}

function getBackups() {
    try {
        $db = new Database();
        $conn = $db->getConnection();
        
        $query = "SELECT * FROM backups ORDER BY created_at DESC";
        $stmt = $conn->query($query);
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        error_log("Get backups error: " . $e->getMessage());
        return [];
    }
}

function deleteBackup($backupName) {
    try {
        // Get the correct backup directory path
        $currentDir = __DIR__; // backend/includes
        $projectRoot = dirname(dirname($currentDir)); // Go up two levels to project root
        $backupPath = $projectRoot . '/backups/' . $backupName;
        
        // Delete from database
        $db = new Database();
        $conn = $db->getConnection();
        
        $query = "DELETE FROM backups WHERE backup_name = ?";
        $stmt = $conn->prepare($query);
        $stmt->execute([$backupName]);
        
        // Delete files
        if (is_dir($backupPath)) {
            deleteDirectory($backupPath);
        }
        
        logSecurityEvent('backup_deleted', "Backup deleted: {$backupName}", 'low');
        
        return true;
    } catch (Exception $e) {
        error_log("Delete backup error: " . $e->getMessage());
        return false;
    }
}

function downloadBackup($backupName) {
    try {
        // Get the correct backup directory path
        $currentDir = __DIR__; // backend/includes
        $projectRoot = dirname(dirname($currentDir)); // Go up two levels to project root
        $backupPath = $projectRoot . '/backups/' . $backupName;
        
        if (!is_dir($backupPath)) {
            return ['success' => false, 'error' => 'بکاپ یافت نشد'];
        }
        
        // Create zip file
        $zipFileName = $backupName . '.zip';
        $zipPath = $projectRoot . '/backups/' . $zipFileName;
        
        $zip = new ZipArchive();
        if ($zip->open($zipPath, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== TRUE) {
            return ['success' => false, 'error' => 'خطا در ایجاد فایل ZIP'];
        }
        
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($backupPath, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::SELF_FIRST
        );
        
        foreach ($iterator as $file) {
            $filePath = $file->getRealPath();
            $relativePath = substr($filePath, strlen($backupPath) + 1);
            
            if ($file->isDir()) {
                $zip->addEmptyDir($relativePath);
            } else {
                $zip->addFile($filePath, $relativePath);
            }
        }
        
        $zip->close();
        
        // Log download event (only if database is available)
        try {
            logSecurityEvent('backup_downloaded', "Backup downloaded: {$backupName}", 'low');
        } catch (Exception $e) {
            // Ignore logging errors
        }
        
        return ['success' => true, 'zip_path' => $zipPath, 'zip_name' => $zipFileName];
    } catch (Exception $e) {
        error_log("Download backup error: " . $e->getMessage());
        return ['success' => false, 'error' => 'خطا در دانلود بکاپ'];
    }
}

function deleteDirectory($path) {
    if (!is_dir($path)) {
        return false;
    }
    
    $iterator = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($path, RecursiveDirectoryIterator::SKIP_DOTS),
        RecursiveIteratorIterator::CHILD_FIRST
    );
    
    foreach ($iterator as $file) {
        if ($file->isDir()) {
            rmdir($file->getRealPath());
        } else {
            unlink($file->getRealPath());
        }
    }
    
    return rmdir($path);
}

function formatFileSize($bytes) {
    $units = ['B', 'KB', 'MB', 'GB', 'TB'];
    
    for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
        $bytes /= 1024;
    }
    
    return round($bytes, 2) . ' ' . $units[$i];
}

function shouldRunAutomaticBackup() {
    try {
        $backupEnabled = getSystemSetting('backup_enabled');
        if ($backupEnabled !== '1') {
            return false;
        }
        
        $frequency = getSystemSetting('backup_frequency');
        $lastBackup = getLastBackupTime();
        
        if (!$lastBackup) {
            return true; // No backup exists, run first backup
        }
        
        $lastBackupTime = strtotime($lastBackup);
        $currentTime = time();
        
        switch ($frequency) {
            case 'daily':
                return ($currentTime - $lastBackupTime) >= 86400; // 24 hours
            case 'weekly':
                return ($currentTime - $lastBackupTime) >= 604800; // 7 days
            case 'monthly':
                return ($currentTime - $lastBackupTime) >= 2592000; // 30 days
            default:
                return false;
        }
    } catch (Exception $e) {
        error_log("Automatic backup check error: " . $e->getMessage());
        return false;
    }
}

function getLastBackupTime() {
    try {
        $db = new Database();
        $conn = $db->getConnection();
        
        $query = "SELECT created_at FROM backups ORDER BY created_at DESC LIMIT 1";
        $stmt = $conn->query($query);
        $result = $stmt->fetch();
        
        return $result ? $result['created_at'] : null;
    } catch (Exception $e) {
        error_log("Get last backup time error: " . $e->getMessage());
        return null;
    }
}

function getCachedIPISP($ipAddress) {
    try {
        // Use ip-api.com to get ISP information
        $url = "http://ip-api.com/json/{$ipAddress}?fields=isp,org";
        $response = file_get_contents($url);
        
        if ($response !== false) {
            $data = json_decode($response, true);
            if ($data && isset($data['isp']) && $data['isp'] !== '') {
                return $data['isp'];
            } elseif ($data && isset($data['org']) && $data['org'] !== '') {
                return $data['org'];
            }
        }
        
        return 'ISP نامشخص';
    } catch (Exception $e) {
        error_log("Error getting ISP info for {$ipAddress}: " . $e->getMessage());
        return 'ISP نامشخص';
    }
}
?>
