<?php
/**
 * IP Check System - بررسی و مسدود کردن IP های مشکوک
 * این فایل باید در ابتدای هر صفحه include شود
 */

// فقط یک بار اجرا شود
if (defined('IP_CHECK_LOADED')) {
    return;
}
define('IP_CHECK_LOADED', true);

// Include required files
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/functions.php';

// Get client IP
$clientIP = getClientIP();

// Skip check for whitelisted IPs
if (isIPWhitelisted($clientIP)) {
    return;
}

// Check if IP is blacklisted
if (isIPBlacklisted($clientIP)) {
    // Log the blocked access
    logSecurityEvent('blocked_access_attempt', 'Blocked IP tried to access: ' . $_SERVER['REQUEST_URI'], 'medium', json_encode([
        'ip' => $clientIP,
        'user_agent' => getUserAgent(),
        'request_uri' => $_SERVER['REQUEST_URI'] ?? '',
        'referer' => $_SERVER['HTTP_REFERER'] ?? ''
    ]));
    
    // Get block details
    $blockDetails = getBlockDetails($clientIP);
    
    // Show blocked page
    http_response_code(403);
    ?>
    <!DOCTYPE html>
    <html lang="fa" dir="rtl">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>دسترسی مسدود شده - گواهیتو</title>
        <link href="https://fonts.googleapis.com/css2?family=Vazirmatn:wght@300;400;500;600;700;800;900&display=swap" rel="stylesheet">
        <style>
            * {
                margin: 0;
                padding: 0;
                box-sizing: border-box;
            }
            
            body {
                font-family: 'Vazirmatn', 'Tahoma', Arial, sans-serif;
                background: linear-gradient(135deg, #1e3c72 0%, #2a5298 100%);
                margin: 0;
                padding: 0;
                min-height: 100vh;
                display: flex;
                align-items: center;
                justify-content: center;
                color: white;
                overflow: hidden;
            }
            
            .background-animation {
                position: fixed;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                background: linear-gradient(45deg, #1e3c72, #2a5298, #1e3c72);
                background-size: 400% 400%;
                animation: gradientShift 15s ease infinite;
                z-index: -1;
            }
            
            @keyframes gradientShift {
                0% { background-position: 0% 50%; }
                50% { background-position: 100% 50%; }
                100% { background-position: 0% 50%; }
            }
            
            .container {
                text-align: center;
                max-width: 600px;
                padding: 3rem 2rem;
                background: rgba(0, 0, 0, 0.4);
                border-radius: 2rem;
                backdrop-filter: blur(20px);
                border: 1px solid rgba(255, 255, 255, 0.1);
                box-shadow: 0 25px 50px rgba(0, 0, 0, 0.3);
                position: relative;
                overflow: hidden;
            }
            
            .container::before {
                content: '';
                position: absolute;
                top: 0;
                left: -100%;
                width: 100%;
                height: 100%;
                background: linear-gradient(90deg, transparent, rgba(255, 255, 255, 0.1), transparent);
                animation: shimmer 3s infinite;
            }
            
            @keyframes shimmer {
                0% { left: -100%; }
                100% { left: 100%; }
            }
            
            .icon {
                font-size: 5rem;
                margin-bottom: 1.5rem;
                animation: pulse 2s infinite;
            }
            
            @keyframes pulse {
                0%, 100% { transform: scale(1); }
                50% { transform: scale(1.1); }
            }
            
            h1 {
                font-size: 2.5rem;
                margin-bottom: 1rem;
                color: #ff6b6b;
                font-weight: 700;
                text-shadow: 0 2px 4px rgba(0, 0, 0, 0.3);
            }
            
            .subtitle {
                font-size: 1.2rem;
                margin-bottom: 2rem;
                color: #ffa726;
                font-weight: 500;
            }
            
            .message {
                font-size: 1.1rem;
                line-height: 1.8;
                margin-bottom: 2rem;
                opacity: 0.9;
                max-width: 500px;
                margin-left: auto;
                margin-right: auto;
            }
            
            .block-info {
                background: rgba(255, 255, 255, 0.1);
                padding: 1.5rem;
                border-radius: 1rem;
                margin: 2rem 0;
                border: 1px solid rgba(255, 255, 255, 0.2);
            }
            
            .ip {
                background: rgba(255, 255, 255, 0.15);
                padding: 0.8rem 1.5rem;
                border-radius: 0.8rem;
                font-family: 'Courier New', monospace;
                font-size: 1.1rem;
                font-weight: 600;
                margin: 1rem 0;
                display: inline-block;
                color: #4caf50;
                border: 1px solid rgba(76, 175, 80, 0.3);
            }
            
            .reason {
                font-size: 1rem;
                color: #ffa726;
                margin: 1rem 0;
                font-weight: 500;
            }
            
            .expires {
                font-size: 0.9rem;
                color: #81c784;
                margin: 1rem 0;
            }
            
            .contact {
                margin-top: 2rem;
                padding-top: 1.5rem;
                border-top: 1px solid rgba(255, 255, 255, 0.2);
                font-size: 1rem;
                opacity: 0.8;
            }
            
            .contact a {
                color: #4fc3f7;
                text-decoration: none;
                font-weight: 500;
            }
            
            .contact a:hover {
                text-decoration: underline;
            }
            
            .contact .email {
                background: rgba(255, 255, 255, 0.1);
                padding: 0.8rem 1.5rem;
                border-radius: 0.8rem;
                display: inline-block;
                margin: 0.5rem 0;
                border: 1px solid rgba(76, 175, 80, 0.3);
            }
            
            .footer {
                position: fixed;
                bottom: 1rem;
                left: 50%;
                transform: translateX(-50%);
                font-size: 0.8rem;
                opacity: 0.6;
                color: #ccc;
            }
            
            @media (max-width: 768px) {
                .container {
                    margin: 1rem;
                    padding: 2rem 1rem;
                    max-width: 95%;
                }
                
                h1 {
                    font-size: 1.8rem;
                }
                
                .subtitle {
                    font-size: 1rem;
                }
                
                .message {
                    font-size: 1rem;
                    line-height: 1.6;
                }
                
                .icon {
                    font-size: 3.5rem;
                }
                
                .block-info {
                    padding: 1rem;
                    margin: 1.5rem 0;
                }
                
                .ip {
                    font-size: 0.9rem;
                    padding: 0.6rem 1rem;
                }
                
                .reason {
                    font-size: 0.9rem;
                }
                
                .expires {
                    font-size: 0.8rem;
                }
                
                .contact {
                    font-size: 0.9rem;
                }
                
                .contact .email {
                    padding: 0.6rem 1rem;
                    font-size: 0.9rem;
                }
                
                .footer {
                    font-size: 0.7rem;
                }
            }
            
            @media (max-width: 480px) {
                .container {
                    margin: 0.5rem;
                    padding: 1.5rem 1rem;
                }
                
                h1 {
                    font-size: 1.5rem;
                }
                
                .icon {
                    font-size: 3rem;
                }
                
                .message {
                    font-size: 0.9rem;
                }
                
                .block-info {
                    padding: 0.8rem;
                }
                
                .ip {
                    font-size: 0.8rem;
                    padding: 0.5rem 0.8rem;
                }
                
                .contact .email {
                    padding: 0.5rem 0.8rem;
                    font-size: 0.8rem;
                }
            }
        </style>
    </head>
    <body>
        <div class="background-animation"></div>
        
        <div class="container">
            <div class="icon">🚫</div>
            <h1>دسترسی مسدود شده</h1>
            <div class="subtitle">Access Denied</div>
            
            <div class="message">
                متأسفانه دسترسی شما به این سایت مسدود شده است.
                <br>این مسدودیت به دلایل امنیتی اعمال شده است.
            </div>
            
            <div class="block-info">
                <div class="ip">IP: <?php echo htmlspecialchars($clientIP); ?></div>
                
                <?php if ($blockDetails): ?>
                    <div class="reason">
                        <strong>دلیل مسدودیت:</strong><br>
                        <?php echo htmlspecialchars($blockDetails['reason']); ?>
                    </div>
                    
                    <?php if ($blockDetails['expires_at']): ?>
                        <div class="expires">
                            <strong>تاریخ انقضا:</strong><br>
                            <?php echo date('Y/m/d H:i', strtotime($blockDetails['expires_at'])); ?>
                            (<?php echo getTimeRemaining($blockDetails['expires_at']); ?>)
                        </div>
                    <?php else: ?>
                        <div class="expires">
                            <strong>نوع مسدودیت:</strong> دائمی
                        </div>
                    <?php endif; ?>
                <?php endif; ?>
            </div>
            
            <div class="contact">
                <strong>برای تماس و رفع مسدودیت:</strong><br>
                <div class="email">
                    📧 <a href="mailto:raytheme.ir@gmail.com">raytheme.ir@gmail.com</a>
                </div>
            </div>
        </div>
        
        <div class="footer">
            © 2024 گواهیتو - سیستم امنیتی پیشرفته
        </div>
    </body>
    </html>
    <?php
    exit();
}

// Rate limiting check (optional - for additional protection)
$rateLimitEnabled = getSystemSetting('enable_rate_limiting', '1');
if ($rateLimitEnabled === '1') {
    if (!checkRateLimit('page_access', 100)) { // 100 requests per hour
        logSecurityEvent('rate_limit_exceeded', 'Rate limit exceeded for page access', 'medium', json_encode([
            'ip' => $clientIP,
            'user_agent' => getUserAgent(),
            'request_uri' => $_SERVER['REQUEST_URI'] ?? ''
        ]));
        
        // Auto-block IP if rate limit exceeded multiple times
        $recentRateLimitEvents = getRecentRateLimitEvents($clientIP, 'page_access', 1); // Last hour
        if (count($recentRateLimitEvents) >= 3) {
            // Auto-block for 24 hours
            autoBlockIP($clientIP, 'Rate limit exceeded multiple times', 24);
        }
        
        http_response_code(429);
        ?>
        <!DOCTYPE html>
        <html lang="fa" dir="rtl">
        <head>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <title>درخواست‌های بیش از حد</title>
            <style>
                body {
                    font-family: 'Vazirmatn', 'Tahoma', Arial, sans-serif;
                    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                    margin: 0;
                    padding: 0;
                    min-height: 100vh;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    color: white;
                }
                .container {
                    text-align: center;
                    max-width: 500px;
                    padding: 2rem;
                    background: rgba(0, 0, 0, 0.3);
                    border-radius: 1rem;
                    backdrop-filter: blur(10px);
                    border: 1px solid rgba(255, 255, 255, 0.1);
                }
                .icon {
                    font-size: 4rem;
                    margin-bottom: 1rem;
                }
                h1 {
                    font-size: 2rem;
                    margin-bottom: 1rem;
                    color: #ffa726;
                }
                p {
                    font-size: 1.1rem;
                    line-height: 1.6;
                    margin-bottom: 1rem;
                    opacity: 0.9;
                }
            </style>
        </head>
        <body>
            <div class="container">
                <div class="icon">⏰</div>
                <h1>درخواست‌های بیش از حد</h1>
                <p>شما درخواست‌های زیادی ارسال کرده‌اید. لطفاً کمی صبر کنید و دوباره تلاش کنید.</p>
                <p>اگر این مشکل ادامه داشت، با مدیر سایت تماس بگیرید.</p>
            </div>
        </body>
        </html>
        <?php
        exit();
    }
}

// Log page access for monitoring
logSecurityEvent('page_access', 'Page accessed', 'low', json_encode([
    'ip' => $clientIP,
    'user_agent' => getUserAgent(),
    'request_uri' => $_SERVER['REQUEST_URI'] ?? '',
    'method' => $_SERVER['REQUEST_METHOD'] ?? 'GET'
]));

// Helper functions
function getBlockDetails($ip) {
    try {
        $db = new Database();
        $conn = $db->getConnection();
        
        $query = "SELECT reason, expires_at, created_at FROM ip_blacklist WHERE ip_address = ? AND (expires_at IS NULL OR expires_at > NOW()) ORDER BY created_at DESC LIMIT 1";
        $stmt = $conn->prepare($query);
        $stmt->execute([$ip]);
        
        return $stmt->fetch();
    } catch (Exception $e) {
        error_log("Block details error: " . $e->getMessage());
        return null;
    }
}

function getTimeRemaining($expiresAt) {
    $expires = strtotime($expiresAt);
    $now = time();
    $diff = $expires - $now;
    
    if ($diff <= 0) {
        return 'منقضی شده';
    }
    
    $hours = floor($diff / 3600);
    $minutes = floor(($diff % 3600) / 60);
    
    if ($hours > 0) {
        return "{$hours} ساعت و {$minutes} دقیقه باقی مانده";
    } else {
        return "{$minutes} دقیقه باقی مانده";
    }
}

function getRecentRateLimitEvents($ip, $endpoint, $hours = 1) {
    try {
        $db = new Database();
        $conn = $db->getConnection();
        
        $query = "SELECT * FROM rate_limit_logs WHERE ip_address = ? AND endpoint = ? AND created_at >= DATE_SUB(NOW(), INTERVAL ? HOUR)";
        $stmt = $conn->prepare($query);
        $stmt->execute([$ip, $endpoint, $hours]);
        
        return $stmt->fetchAll();
    } catch (Exception $e) {
        error_log("Rate limit events error: " . $e->getMessage());
        return [];
    }
}

function autoBlockIP($ip, $reason, $hours = 24) {
    try {
        $db = new Database();
        $conn = $db->getConnection();
        
        $expiresAt = date('Y-m-d H:i:s', strtotime("+$hours hours"));
        $query = "INSERT INTO ip_blacklist (ip_address, reason, expires_at, added_by, created_at) VALUES (?, ?, ?, 1, NOW()) ON DUPLICATE KEY UPDATE reason = ?, expires_at = ?, updated_at = NOW()";
        $stmt = $conn->prepare($query);
        $stmt->execute([$ip, $reason, $expiresAt, $reason, $expiresAt]);
        
        logSecurityEvent('ip_auto_blocked', 'IP auto-blocked by system', 'medium', json_encode([
            'ip' => $ip,
            'reason' => $reason,
            'duration' => $hours
        ]));
        
        return true;
    } catch (Exception $e) {
        error_log("Auto block IP error: " . $e->getMessage());
        return false;
    }
}
?>
