<?php
// Set proper charset headers
header('Content-Type: text/html; charset=utf-8');
mb_internal_encoding('UTF-8');
mb_http_output('UTF-8');

// Start session
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once 'config/database.php';
require_once 'includes/functions.php';

// Check authentication
if (!isLoggedIn()) {
    logSecurityEvent('unauthorized_access_attempt', 'Unauthorized access attempt to security.php from IP: ' . getClientIP(), 'high');
    header('Location: 403.php');
    exit;
}
checkSessionTimeout();

$message = '';
$error = '';

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    try {
        $db = new Database();
        $conn = $db->getConnection();
        
        switch ($action) {
            case 'clear_logs':
                $days = (int)($_POST['days'] ?? 30);
                $severity = $_POST['severity'] ?? '';
                
                // Build query based on severity filter
                if (!empty($severity)) {
                    $deleteQuery = "DELETE FROM security_logs WHERE created_at < DATE_SUB(NOW(), INTERVAL ? DAY) AND severity = ?";
                    $deleteStmt = $conn->prepare($deleteQuery);
                    $deleteStmt->execute([$days, $severity]);
                    $message = "لاگ‌های امنیتی با سطح اهمیت '$severity' قدیمی‌تر از $days روز با موفقیت پاک شد";
                } else {
                    $deleteQuery = "DELETE FROM security_logs WHERE created_at < DATE_SUB(NOW(), INTERVAL ? DAY)";
                    $deleteStmt = $conn->prepare($deleteQuery);
                    $deleteStmt->execute([$days]);
                    $message = "لاگ‌های امنیتی قدیمی‌تر از $days روز با موفقیت پاک شد";
                }
                
                logSecurityEvent('security_logs_cleared', 'Days: ' . $days . ', Severity: ' . ($severity ?: 'all'), 'low');
                break;
                
            case 'clear_rate_limits':
                $deleteQuery = "DELETE FROM rate_limit_logs WHERE created_at < DATE_SUB(NOW(), INTERVAL 1 HOUR)";
                $conn->exec($deleteQuery);
                
                $message = "لاگ‌های محدودیت نرخ با موفقیت پاک شد";
                logSecurityEvent('rate_limit_logs_cleared', '', 'low');
                break;
                
            case 'clear_today_logs':
                $severity = $_POST['severity'] ?? '';
                
                // Build query based on severity filter
                if (!empty($severity)) {
                    $deleteQuery = "DELETE FROM security_logs WHERE DATE(created_at) = CURDATE() AND severity = ?";
                    $deleteStmt = $conn->prepare($deleteQuery);
                    $deleteStmt->execute([$severity]);
                    $message = "لاگ‌های امنیتی امروز با سطح اهمیت '$severity' با موفقیت پاک شد";
                } else {
                    $deleteQuery = "DELETE FROM security_logs WHERE DATE(created_at) = CURDATE()";
                    $conn->exec($deleteQuery);
                    $message = "لاگ‌های امنیتی امروز با موفقیت پاک شد";
                }
                
                logSecurityEvent('today_logs_cleared', 'Severity: ' . ($severity ?: 'all'), 'low');
                break;
                
            case 'clear_all_logs':
                $severity = $_POST['severity'] ?? '';
                
                // Build query based on severity filter
                if (!empty($severity)) {
                    $deleteQuery = "DELETE FROM security_logs WHERE severity = ?";
                    $deleteStmt = $conn->prepare($deleteQuery);
                    $deleteStmt->execute([$severity]);
                    $message = "همه لاگ‌های امنیتی با سطح اهمیت '$severity' با موفقیت پاک شد";
                } else {
                    $deleteQuery = "DELETE FROM security_logs";
                    $conn->exec($deleteQuery);
                    $message = "همه لاگ‌های امنیتی با موفقیت پاک شد";
                }
                
                logSecurityEvent('all_logs_cleared', 'Severity: ' . ($severity ?: 'all'), 'high');
                break;
                
            case 'add_blacklist':
                $ip = sanitizeInput($_POST['ip'] ?? '');
                $reason = sanitizeInput($_POST['reason'] ?? '');
                $expires = $_POST['expires'] ?? '';
                
                if (filter_var($ip, FILTER_VALIDATE_IP)) {
                    $expiresAt = $expires ? $expires : null;
                    
                    // Get admin user ID for foreign key constraint
                    $adminStmt = $conn->query("SELECT id FROM admin_users LIMIT 1");
                    $adminUser = $adminStmt->fetch(PDO::FETCH_ASSOC);
                    $adminId = $adminUser ? $adminUser['id'] : 1;
                    
                    $insertQuery = "INSERT INTO ip_blacklist (ip_address, reason, expires_at, added_by) VALUES (?, ?, ?, ?) ON DUPLICATE KEY UPDATE reason = ?, expires_at = ?, updated_at = NOW()";
                    $insertStmt = $conn->prepare($insertQuery);
                    $insertStmt->execute([$ip, $reason, $expiresAt, $adminId, $reason, $expiresAt]);
                    
                    $message = "آدرس IP $ip به لیست سیاه اضافه شد";
                    logSecurityEvent('ip_blacklisted', 'IP: ' . $ip . ', Reason: ' . $reason, 'medium');
                } else {
                    $error = 'آدرس IP نامعتبر است';
                }
                break;
                
            case 'remove_blacklist':
                $ip = sanitizeInput($_POST['ip'] ?? '');
                
                $deleteQuery = "DELETE FROM ip_blacklist WHERE ip_address = ?";
                $deleteStmt = $conn->prepare($deleteQuery);
                $deleteStmt->execute([$ip]);
                
                $message = "آدرس IP $ip از لیست سیاه حذف شد";
                logSecurityEvent('ip_unblacklisted', 'IP: ' . $ip, 'medium');
                break;
                
            case 'add_whitelist':
                $ip = sanitizeInput($_POST['ip'] ?? '');
                $description = sanitizeInput($_POST['description'] ?? '');
                
                if (filter_var($ip, FILTER_VALIDATE_IP)) {
                    // Get admin user ID for foreign key constraint
                    $adminStmt = $conn->query("SELECT id FROM admin_users LIMIT 1");
                    $adminUser = $adminStmt->fetch(PDO::FETCH_ASSOC);
                    $adminId = $adminUser ? $adminUser['id'] : 1;
                    
                    $insertQuery = "INSERT INTO ip_whitelist (ip_address, description, added_by) VALUES (?, ?, ?) ON DUPLICATE KEY UPDATE description = ?, updated_at = NOW()";
                    $insertStmt = $conn->prepare($insertQuery);
                    $insertStmt->execute([$ip, $description, $adminId, $description]);
                    
                    $message = "آدرس IP $ip به لیست سفید اضافه شد";
                    logSecurityEvent('ip_whitelisted', 'IP: ' . $ip . ', Description: ' . $description, 'low');
                } else {
                    $error = 'آدرس IP نامعتبر است';
                }
                break;
                
            case 'remove_whitelist':
                $ip = sanitizeInput($_POST['ip'] ?? '');
                
                $deleteQuery = "DELETE FROM ip_whitelist WHERE ip_address = ?";
                $deleteStmt = $conn->prepare($deleteQuery);
                $deleteStmt->execute([$ip]);
                
                $message = "آدرس IP $ip از لیست سفید حذف شد";
                logSecurityEvent('ip_unwhitelisted', 'IP: ' . $ip, 'low');
                break;
                
            case 'set_admin_ip_restriction':
                $ip = sanitizeInput($_POST['ip'] ?? '');
                $adminEmail = sanitizeInput($_POST['admin_email'] ?? '');
                
                if (filter_var($ip, FILTER_VALIDATE_IP) && validateEmail($adminEmail)) {
                    if (setAdminIPRestriction($ip, $adminEmail)) {
                        $message = "محدودیت IP ادمین تنظیم شد. فقط IP $ip می‌تواند به پنل دسترسی داشته باشد.";
                    } else {
                        $error = 'خطا در تنظیم محدودیت IP';
                    }
                } else {
                    $error = 'آدرس IP یا ایمیل نامعتبر است';
                }
                break;
                
            case 'remove_admin_ip_restriction':
                if (removeAdminIPRestriction()) {
                    $message = "محدودیت IP ادمین برداشته شد. حالا همه IP ها می‌توانند به پنل دسترسی داشته باشند.";
                } else {
                    $error = 'خطا در برداشتن محدودیت IP';
                }
                break;
        }
    } catch (Exception $e) {
        error_log("Security page error: " . $e->getMessage());
        $error = 'خطا در عملیات. لطفاً دوباره تلاش کنید';
    }
}

// Get security data
try {
    $db = new Database();
    $conn = $db->getConnection();
    
    // Get recent security events
    $eventsQuery = "SELECT * FROM security_logs ORDER BY created_at DESC LIMIT 50";
    $eventsStmt = $conn->query($eventsQuery);
    $events = $eventsStmt->fetchAll();
    
    // Get blacklisted IPs
    $blacklistQuery = "SELECT * FROM ip_blacklist ORDER BY created_at DESC";
    $blacklistStmt = $conn->query($blacklistQuery);
    $blacklist = $blacklistStmt->fetchAll();
    
    // Get whitelisted IPs
    $whitelistQuery = "SELECT * FROM ip_whitelist WHERE is_active = 1 ORDER BY created_at DESC";
    $whitelistStmt = $conn->query($whitelistQuery);
    $whitelist = $whitelistStmt->fetchAll();
    
    // Get statistics
    $statsQuery = "SELECT 
        COUNT(*) as total_events,
        SUM(CASE WHEN severity = 'critical' THEN 1 ELSE 0 END) as critical_events,
        SUM(CASE WHEN severity = 'high' THEN 1 ELSE 0 END) as high_events,
        SUM(CASE WHEN severity = 'medium' THEN 1 ELSE 0 END) as medium_events,
        SUM(CASE WHEN severity = 'low' THEN 1 ELSE 0 END) as low_events,
        SUM(CASE WHEN DATE(created_at) = CURDATE() THEN 1 ELSE 0 END) as today_events
        FROM security_logs";
    $statsStmt = $conn->query($statsQuery);
    $stats = $statsStmt->fetch();
    
    // Get event types
    $eventTypesQuery = "SELECT event_type, COUNT(*) as count FROM security_logs GROUP BY event_type ORDER BY count DESC LIMIT 10";
    $eventTypesStmt = $conn->query($eventTypesQuery);
    $eventTypes = $eventTypesStmt->fetchAll();
    
} catch (Exception $e) {
    error_log("Security data error: " . $e->getMessage());
    $error = 'خطا در بارگذاری اطلاعات امنیتی';
    $events = [];
    $blacklist = [];
    $whitelist = [];
    $stats = ['total_events' => 0, 'critical_events' => 0, 'high_events' => 0, 'medium_events' => 0, 'low_events' => 0, 'today_events' => 0];
    $eventTypes = [];
}
?>
<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>امنیت سیستم - پنل مدیریت گواهیتو</title>
    <link rel="stylesheet" href="assets/admin-style.css">
    <link rel="icon" type="image/png" href="assets/images/logo.png">
    <?php include 'includes/admin-styles.php'; ?>
</head>
<body>
    <?php include 'includes/header.php'; ?>
    <?php include 'includes/sidebar.php'; ?>
    
    <!-- Sidebar Overlay for Mobile -->
    <div class="sidebar-overlay" id="sidebarOverlay"></div>

    <!-- Main Content -->
    <div class="main-content" id="mainContent">
        <div class="page-header">
            <h1 class="page-title">🔒 امنیت سیستم</h1>
            <p class="page-description">مدیریت امنیت و نظارت بر رویدادهای سیستم</p>
        </div>

        <?php if ($error): ?>
            <div class="alert alert-danger">
                <?php echo htmlspecialchars($error); ?>
            </div>
        <?php endif; ?>

        <?php if ($message): ?>
            <div class="alert alert-success">
                <?php echo htmlspecialchars($message); ?>
            </div>
        <?php endif; ?>

        <!-- Security Statistics -->
        <div class="grid grid-cols-1 grid-cols-2 grid-cols-3 grid-cols-4" style="margin-bottom: 2rem;">
            <div class="stats-card">
                <div class="stats-number"><?php echo $stats['total_events']; ?></div>
                <div class="stats-label">کل رویدادها</div>
            </div>
            
            <div class="stats-card">
                <div class="stats-number"><?php echo $stats['today_events']; ?></div>
                <div class="stats-label">رویدادهای امروز</div>
            </div>
            
            <div class="stats-card">
                <div class="stats-number"><?php echo $stats['critical_events']; ?></div>
                <div class="stats-label">رویدادهای بحرانی</div>
            </div>
            
            <div class="stats-card">
                <div class="stats-number"><?php echo $stats['high_events']; ?></div>
                <div class="stats-label">رویدادهای مهم</div>
            </div>
            
            <div class="stats-card">
                <div class="stats-number"><?php echo count($blacklist); ?></div>
                <div class="stats-label">IP های مسدود</div>
            </div>
            
            <div class="stats-card">
                <div class="stats-number"><?php echo count($whitelist); ?></div>
                <div class="stats-label">IP های مجاز</div>
            </div>
        </div>

        <!-- Security Actions -->
        <div class="grid grid-cols-1 grid-cols-2 grid-cols-3 grid-cols-4" style="gap: 2rem; margin-bottom: 2rem;">
            <!-- Clear Logs -->
            <div class="card">
                <div class="card-header">
                    <h2 class="card-title">🗑️ پاک کردن لاگ‌ها</h2>
                </div>
                
                <form method="POST" action="">
                    <input type="hidden" name="action" value="clear_logs">
                    
                    <div class="form-group">
                        <label for="days" class="form-label">پاک کردن لاگ‌های قدیمی‌تر از</label>
                        <select id="days" name="days" class="form-select">
                            <option value="1">1 روز (دیروز)</option>
                            <option value="7">7 روز (هفته گذشته)</option>
                            <option value="30" selected>30 روز (ماه گذشته)</option>
                            <option value="90">90 روز (3 ماه گذشته)</option>
                            <option value="180">180 روز (6 ماه گذشته)</option>
                            <option value="365">365 روز (سال گذشته)</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label for="severity" class="form-label">فقط لاگ‌های با سطح اهمیت</label>
                        <select id="severity" name="severity" class="form-select">
                            <option value="">همه سطوح</option>
                            <option value="low">کم</option>
                            <option value="medium">متوسط</option>
                            <option value="high">زیاد</option>
                            <option value="critical">بحرانی</option>
                        </select>
                    </div>
                    
                    <button type="submit" class="btn btn-warning" onclick="return confirm('آیا از پاک کردن لاگ‌های قدیمی اطمینان دارید؟ این عملیات قابل بازگشت نیست.')">
                        🗑️ پاک کردن لاگ‌ها
                    </button>
                </form>
            </div>

            <!-- Clear Today's Logs -->
            <div class="card">
                <div class="card-header">
                    <h2 class="card-title">📅 پاک کردن لاگ‌های امروز</h2>
                </div>
                
                <form method="POST" action="">
                    <input type="hidden" name="action" value="clear_today_logs">
                    
                    <div class="form-group">
                        <label for="today_severity" class="form-label">فقط لاگ‌های با سطح اهمیت</label>
                        <select id="today_severity" name="severity" class="form-select">
                            <option value="">همه سطوح</option>
                            <option value="low">کم</option>
                            <option value="medium">متوسط</option>
                            <option value="high">زیاد</option>
                            <option value="critical">بحرانی</option>
                        </select>
                    </div>
                    
                    <button type="submit" class="btn btn-danger" onclick="return confirm('آیا از پاک کردن لاگ‌های امروز اطمینان دارید؟ این عملیات قابل بازگشت نیست.')">
                        📅 پاک کردن لاگ‌های امروز
                    </button>
                </form>
            </div>

            <!-- Clear Rate Limits -->
            <div class="card">
                <div class="card-header">
                    <h2 class="card-title">⚡ پاک کردن محدودیت‌های نرخ</h2>
                </div>
                
                <form method="POST" action="">
                    <input type="hidden" name="action" value="clear_rate_limits">
                    
                    <p style="color: var(--dark-300); margin-bottom: 1rem;">
                        پاک کردن لاگ‌های محدودیت نرخ قدیمی‌تر از 1 ساعت
                    </p>
                    
                    <button type="submit" class="btn btn-info" onclick="return confirm('آیا از پاک کردن لاگ‌های محدودیت نرخ اطمینان دارید؟')">
                        پاک کردن محدودیت‌ها
                    </button>
                </form>
            </div>

            <!-- Clear All Logs -->
            <div class="card">
                <div class="card-header">
                    <h2 class="card-title">💥 پاک کردن همه لاگ‌ها</h2>
                </div>
                
                <form method="POST" action="">
                    <input type="hidden" name="action" value="clear_all_logs">
                    
                    <div class="form-group">
                        <label for="all_severity" class="form-label">فقط لاگ‌های با سطح اهمیت</label>
                        <select id="all_severity" name="severity" class="form-select">
                            <option value="">همه سطوح</option>
                            <option value="low">کم</option>
                            <option value="medium">متوسط</option>
                            <option value="high">زیاد</option>
                            <option value="critical">بحرانی</option>
                        </select>
                    </div>
                    
                    <button type="submit" class="btn btn-danger" onclick="return confirm('⚠️ هشدار: این عملیات همه لاگ‌های امنیتی را پاک می‌کند. آیا از این کار اطمینان دارید؟ این عملیات قابل بازگشت نیست.')">
                        💥 پاک کردن همه لاگ‌ها
                    </button>
                </form>
            </div>
        </div>

        <!-- Admin IP Restriction -->
        <div class="card" style="margin-bottom: 2rem;">
            <div class="card-header">
                <h2 class="card-title">🔒 محدودیت IP ادمین</h2>
            </div>
            
            <?php 
            $currentRestriction = getCurrentAdminIPRestriction();
            $isRestrictionEnabled = isAdminIPRestrictionEnabled();
            ?>
            
            <?php if ($isRestrictionEnabled && $currentRestriction): ?>
                <div class="alert alert-warning" style="margin-bottom: 1rem;">
                    <strong>⚠️ محدودیت IP فعال است:</strong><br>
                    فقط IP <code><?php echo htmlspecialchars($currentRestriction['allowed_ip']); ?></code> می‌تواند به پنل ادمین دسترسی داشته باشد.<br>
                    ایمیل: <?php echo htmlspecialchars($currentRestriction['admin_email']); ?>
                </div>
                
                <form method="POST" action="" style="margin-bottom: 1rem;">
                    <input type="hidden" name="action" value="remove_admin_ip_restriction">
                    <button type="submit" class="btn btn-warning" onclick="return confirm('آیا از برداشتن محدودیت IP اطمینان دارید؟')">
                        برداشتن محدودیت IP
                    </button>
                </form>
                
            <?php else: ?>
                <p style="color: var(--dark-300); margin-bottom: 1rem;">
                    در حال حاضر محدودیت IP فعال نیست. همه IP ها می‌توانند به پنل ادمین دسترسی داشته باشند.
                </p>
                
                <form method="POST" action="" style="margin-bottom: 1rem;">
                    <input type="hidden" name="action" value="set_admin_ip_restriction">
                    
                    <div class="form-group">
                        <label for="restriction_ip" class="form-label">آدرس IP مجاز</label>
                        <input type="text" id="restriction_ip" name="ip" class="form-input" 
                               placeholder="192.168.1.1" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="restriction_email" class="form-label">ایمیل ادمین</label>
                        <input type="email" id="restriction_email" name="admin_email" class="form-input" 
                               placeholder="ایمیل ادمین" required>
                    </div>
                    
                    <button type="submit" class="btn btn-danger" onclick="return confirm('آیا از تنظیم محدودیت IP اطمینان دارید؟ این کار دسترسی سایر IP ها را مسدود می‌کند.')">
                        تنظیم محدودیت IP
                    </button>
                </form>
            <?php endif; ?>
            
                         <div style="margin-top: 1rem; padding-top: 1rem; border-top: 1px solid var(--border-color);">
                 <p style="font-size: 0.9rem; color: var(--dark-400);">
                     <strong>نکته:</strong> در صورت فراموشی IP یا نیاز به دسترسی اضطراری، از 
                     <a href="Hua89fe754tr7gyu7o9werdhy79erwte0gw04etyg.php" style="color: #e74c3c; text-decoration: none;">صفحه دسترسی اضطراری</a> استفاده کنید.
                 </p>
             </div>
        </div>

        <!-- IP Management -->
        <div class="grid grid-cols-1 grid-cols-2" style="gap: 2rem; margin-bottom: 2rem;">
            <!-- Blacklist Management -->
            <div class="card">
                <div class="card-header">
                    <h2 class="card-title">🚫 مدیریت لیست سیاه</h2>
                </div>
                
                <form method="POST" action="" style="margin-bottom: 1rem;">
                    <input type="hidden" name="action" value="add_blacklist">
                    
                    <div class="form-group">
                        <label for="blacklist_ip" class="form-label">آدرس IP</label>
                        <input type="text" id="blacklist_ip" name="ip" class="form-input" placeholder="192.168.1.1" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="blacklist_reason" class="form-label">دلیل</label>
                        <input type="text" id="blacklist_reason" name="reason" class="form-input" placeholder="دلیل مسدودیت">
                    </div>
                    
                    <div class="form-group">
                        <label for="blacklist_expires" class="form-label">تاریخ انقضا (اختیاری)</label>
                        <input type="datetime-local" id="blacklist_expires" name="expires" class="form-input">
                    </div>
                    
                    <button type="submit" class="btn btn-danger">اضافه به لیست سیاه</button>
                </form>
                
                <?php if (!empty($blacklist)): ?>
                    <div class="table-container">
                        <table class="table">
                            <thead>
                                <tr>
                                    <th>آدرس IP</th>
                                    <th>دلیل</th>
                                    <th>تاریخ انقضا</th>
                                    <th>عملیات</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($blacklist as $ip): ?>
                                    <tr>
                                        <td><code><?php echo htmlspecialchars($ip['ip_address']); ?></code></td>
                                        <td><?php echo htmlspecialchars($ip['reason'] ?? '-'); ?></td>
                                        <td>
                                            <?php 
                                            if ($ip['expires_at']) {
                                                echo formatDate($ip['expires_at']);
                                            } else {
                                                echo 'نامحدود';
                                            }
                                            ?>
                                        </td>
                                        <td>
                                            <form method="POST" action="" style="display: inline;">
                                                <input type="hidden" name="action" value="remove_blacklist">
                                                <input type="hidden" name="ip" value="<?php echo htmlspecialchars($ip['ip_address']); ?>">
                                                <button type="submit" class="btn btn-success btn-sm" onclick="return confirm('آیا از حذف این IP از لیست سیاه اطمینان دارید؟')">
                                                    حذف
                                                </button>
                                            </form>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php else: ?>
                    <p style="color: var(--dark-400); text-align: center; padding: 1rem;">هیچ IP مسدودی یافت نشد</p>
                <?php endif; ?>
            </div>

            <!-- Whitelist Management -->
            <div class="card">
                <div class="card-header">
                    <h2 class="card-title">✅ مدیریت لیست سفید</h2>
                </div>
                
                <form method="POST" action="" style="margin-bottom: 1rem;">
                    <input type="hidden" name="action" value="add_whitelist">
                    
                    <div class="form-group">
                        <label for="whitelist_ip" class="form-label">آدرس IP</label>
                        <input type="text" id="whitelist_ip" name="ip" class="form-input" placeholder="192.168.1.1" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="whitelist_description" class="form-label">توضیحات</label>
                        <input type="text" id="whitelist_description" name="description" class="form-input" placeholder="توضیحات">
                    </div>
                    
                    <button type="submit" class="btn btn-success">اضافه به لیست سفید</button>
                </form>
                
                <?php if (!empty($whitelist)): ?>
                    <div class="table-container">
                        <table class="table">
                            <thead>
                                <tr>
                                    <th>آدرس IP</th>
                                    <th>توضیحات</th>
                                    <th>تاریخ اضافه</th>
                                    <th>عملیات</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($whitelist as $ip): ?>
                                    <tr>
                                        <td><code><?php echo htmlspecialchars($ip['ip_address']); ?></code></td>
                                        <td><?php echo htmlspecialchars($ip['description'] ?? '-'); ?></td>
                                        <td><?php echo formatDate($ip['created_at']); ?></td>
                                        <td>
                                            <form method="POST" action="" style="display: inline;">
                                                <input type="hidden" name="action" value="remove_whitelist">
                                                <input type="hidden" name="ip" value="<?php echo htmlspecialchars($ip['ip_address']); ?>">
                                                <button type="submit" class="btn btn-danger btn-sm" onclick="return confirm('آیا از حذف این IP از لیست سفید اطمینان دارید؟')">
                                                    حذف
                                                </button>
                                            </form>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php else: ?>
                    <p style="color: var(--dark-400); text-align: center; padding: 1rem;">هیچ IP مجازی یافت نشد</p>
                <?php endif; ?>
            </div>
        </div>

        <!-- Security Events -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">📊 رویدادهای امنیتی اخیر</h2>
            </div>
            
            <?php if (!empty($events)): ?>
                <div class="table-container">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>نوع رویداد</th>
                                <th>آدرس IP</th>
                                <th>سطح اهمیت</th>
                                <th>جزئیات</th>
                                <th>تاریخ</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($events as $event): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($event['event_type']); ?></td>
                                    <td><code><?php echo htmlspecialchars($event['ip_address']); ?></code></td>
                                    <td>
                                        <?php
                                        $severityClass = match($event['severity']) {
                                            'low' => 'badge-info',
                                            'medium' => 'badge-warning',
                                            'high' => 'badge-danger',
                                            'critical' => 'badge-danger',
                                            default => 'badge-info'
                                        };
                                        ?>
                                        <span class="badge <?php echo $severityClass; ?>">
                                            <?php echo htmlspecialchars($event['severity']); ?>
                                        </span>
                                    </td>
                                    <td><?php echo htmlspecialchars($event['details']); ?></td>
                                    <td><?php echo formatDate($event['created_at']); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php else: ?>
                <p style="color: var(--dark-400); text-align: center; padding: 2rem;">هیچ رویداد امنیتی‌ای یافت نشد</p>
            <?php endif; ?>
        </div>

        <!-- Event Types Statistics -->
        <?php if (!empty($eventTypes)): ?>
            <div class="card">
                <div class="card-header">
                    <h2 class="card-title">📈 آمار انواع رویدادها</h2>
                </div>
                
                <div class="table-container">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>نوع رویداد</th>
                                <th>تعداد</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($eventTypes as $type): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($type['event_type']); ?></td>
                                    <td><?php echo $type['count']; ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        <?php endif; ?>
    </div>

    <script>
        // Form validation
        document.querySelectorAll('form').forEach(form => {
            form.addEventListener('submit', function(e) {
                const ipInputs = form.querySelectorAll('input[name="ip"]');
                ipInputs.forEach(input => {
                    if (input.value && !isValidIP(input.value)) {
                        e.preventDefault();
                        alert('لطفاً آدرس IP معتبر وارد کنید');
                        input.focus();
                        return false;
                    }
                });
            });
        });
        
        // IP validation function
        function isValidIP(ip) {
            const ipRegex = /^(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)$/;
            return ipRegex.test(ip);
        }
        
        // Auto-refresh page every 30 seconds
        setTimeout(function() {
            location.reload();
        }, 30000);
    </script>
</body>
</html>
