<?php
// Set proper charset headers
header('Content-Type: text/html; charset=utf-8');
mb_internal_encoding('UTF-8');
mb_http_output('UTF-8');

// Start session
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once 'config/database.php';
require_once 'includes/functions.php';

// Check authentication
if (!isLoggedIn()) {
    logSecurityEvent('unauthorized_access_attempt', 'Unauthorized access attempt to settings.php from IP: ' . getClientIP(), 'high');
    header('Location: 403.php');
    exit;
}
checkSessionTimeout();

$message = '';
$error = '';

// Handle settings update
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    try {
        $db = new Database();
        $conn = $db->getConnection();
        
        switch ($action) {
            case 'update_settings':
                // Handle favicon upload first
                if (isset($_FILES['favicon']) && $_FILES['favicon']['error'] === UPLOAD_ERR_OK) {
                    $allowedTypes = ['image/x-icon', 'image/png', 'image/jpeg', 'image/gif'];
                    $maxSize = 1024 * 1024; // 1MB
                    
                    if (!in_array($_FILES['favicon']['type'], $allowedTypes)) {
                        $error = 'نوع فایل نامعتبر است. فقط فایل‌های ICO، PNG، JPEG و GIF مجاز هستند.';
                        break;
                    }
                    
                    if ($_FILES['favicon']['size'] > $maxSize) {
                        $error = 'حجم فایل نباید بیشتر از 1 مگابایت باشد.';
                        break;
                    }
                    
                    $uploadDir = 'assets/favicons/';
                    if (!is_dir($uploadDir)) {
                        mkdir($uploadDir, 0755, true);
                    }
                    
                    $fileName = 'favicon.' . pathinfo($_FILES['favicon']['name'], PATHINFO_EXTENSION);
                    $filePath = $uploadDir . $fileName;
                    
                    if (move_uploaded_file($_FILES['favicon']['tmp_name'], $filePath)) {
                        setSystemSetting('favicon_path', $filePath);
                        $message = "فاوآیکون با موفقیت آپلود شد و ";
                        logSecurityEvent('favicon_uploaded', 'Uploaded by: ' . $_SESSION['admin_username'], 'low');
                    } else {
                        $error = 'خطا در آپلود فایل';
                        break;
                    }
                }
                
                // Handle other settings
                $settings = [
                    'rate_limit_per_hour' => (int)($_POST['rate_limit_per_hour'] ?? 5),
                    'session_timeout_minutes' => (int)($_POST['session_timeout_minutes'] ?? 30),
                    'enable_xss_protection' => (int)($_POST['enable_xss_protection'] ?? 1),
                    'enable_sql_injection_protection' => (int)($_POST['enable_sql_injection_protection'] ?? 1),
                    'enable_recaptcha' => (int)($_POST['enable_recaptcha'] ?? 1),
                    'admin_email' => sanitizeInput($_POST['admin_email'] ?? ''),
                    'site_name' => sanitizeInput($_POST['site_name'] ?? ''),
                    'site_url' => sanitizeInput($_POST['site_url'] ?? ''),
                    'maintenance_mode' => (int)($_POST['maintenance_mode'] ?? 0),
                    'maintenance_message' => sanitizeInput($_POST['maintenance_message'] ?? ''),
                    'backup_enabled' => (int)($_POST['backup_enabled'] ?? 0),
                    'backup_frequency' => sanitizeInput($_POST['backup_frequency'] ?? 'daily'),
                    'log_retention_days' => (int)($_POST['log_retention_days'] ?? 30),
                    'max_login_attempts' => (int)($_POST['max_login_attempts'] ?? 5),
                    'lockout_duration_minutes' => (int)($_POST['lockout_duration_minutes'] ?? 15),
                    'password_min_length' => (int)($_POST['password_min_length'] ?? 8),
                    'require_strong_password' => (int)($_POST['require_strong_password'] ?? 1),
                    'twofa_enabled' => (int)($_POST['twofa_enabled'] ?? 0),
                    'twofa_code_expiry' => (int)($_POST['twofa_code_expiry'] ?? 300),
                    'email_notifications' => (int)($_POST['email_notifications'] ?? 1),
                    'auto_logout_minutes' => (int)($_POST['auto_logout_minutes'] ?? 30)
                ];
                
                foreach ($settings as $key => $value) {
                    setSystemSetting($key, $value);
                }
                
                $message .= "تنظیمات با موفقیت بروزرسانی شد";
                logSecurityEvent('settings_updated', 'Updated by: ' . $_SESSION['admin_username'], 'low');
                break;
                
            case 'create_backup':
                $backupType = $_POST['backup_type'] ?? 'full';
                $result = createBackup($backupType);
                
                if ($result['success']) {
                    $message = "بکاپ با موفقیت ایجاد شد: " . $result['backup_name'];
                } else {
                    $error = "خطا در ایجاد بکاپ: " . $result['error'];
                }
                break;
                
            case 'reset_settings':
                $defaultSettings = [
                    'rate_limit_per_hour' => '5',
                    'session_timeout_minutes' => '30',
                    'enable_xss_protection' => '1',
                    'enable_sql_injection_protection' => '1',
                    'enable_recaptcha' => '1',
                    'admin_email' => 'info@gavahito.ir',
                    'site_name' => 'Gavahito Admin Panel',
                    'site_url' => 'https://gavahito.ir/',
                    'maintenance_mode' => '0',
                    'maintenance_message' => 'سیستم در حال نگهداری است. لطفاً کمی صبر کنید.',
                    'backup_enabled' => '0',
                    'backup_frequency' => 'daily',
                    'log_retention_days' => '30',
                    'max_login_attempts' => '5',
                    'lockout_duration_minutes' => '15',
                    'password_min_length' => '8',
                    'require_strong_password' => '1',
                    'twofa_enabled' => '0',
                    'twofa_code_expiry' => '300',
                    'email_notifications' => '1',
                    'auto_logout_minutes' => '30'
                ];
                
                foreach ($defaultSettings as $key => $value) {
                    setSystemSetting($key, $value);
                }
                
                $message = "تنظیمات به حالت پیش‌فرض بازگردانده شد";
                logSecurityEvent('settings_reset', 'Reset by: ' . $_SESSION['admin_username'], 'medium');
                break;
                
            case 'export_settings':
                $settings = [];
                $query = "SELECT setting_key, setting_value FROM system_settings";
                $stmt = $conn->query($query);
                while ($row = $stmt->fetch()) {
                    $settings[$row['setting_key']] = $row['setting_value'];
                }
                
                header('Content-Type: application/json');
                header('Content-Disposition: attachment; filename="gavahito_settings_' . date('Y-m-d_H-i-s') . '.json"');
                echo json_encode($settings, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
                exit();
                break;
                
            case 'import_settings':
                if (isset($_FILES['settings_file']) && $_FILES['settings_file']['error'] === UPLOAD_ERR_OK) {
                    $content = file_get_contents($_FILES['settings_file']['tmp_name']);
                    $settings = json_decode($content, true);
                    
                    if (json_last_error() === JSON_ERROR_NONE && is_array($settings)) {
                        foreach ($settings as $key => $value) {
                            setSystemSetting($key, $value);
                        }
                        
                        $message = "تنظیمات با موفقیت وارد شد";
                        logSecurityEvent('settings_imported', 'Imported by: ' . $_SESSION['admin_username'], 'medium');
                    } else {
                        $error = 'فایل تنظیمات نامعتبر است';
                    }
                } else {
                    $error = 'خطا در آپلود فایل';
                }
                break;
        }
    } catch (Exception $e) {
        error_log("Settings error: " . $e->getMessage());
        $error = 'خطا در عملیات. لطفاً دوباره تلاش کنید';
    }
}

// Get current settings
$currentSettings = [];
try {
    $db = new Database();
    $conn = $db->getConnection();
    
    $query = "SELECT setting_key, setting_value FROM system_settings";
    $stmt = $conn->query($query);
    while ($row = $stmt->fetch()) {
        $currentSettings[$row['setting_key']] = $row['setting_value'];
    }
} catch (Exception $e) {
    error_log("Settings load error: " . $e->getMessage());
    $error = 'خطا در بارگذاری تنظیمات';
}

// Get current favicon path
$currentFavicon = $currentSettings['favicon_path'] ?? '';

// Set default values if not exists
$defaults = [
    'rate_limit_per_hour' => '5',
    'session_timeout_minutes' => '30',
    'enable_xss_protection' => '1',
    'enable_sql_injection_protection' => '1',
    'enable_recaptcha' => '1',
    'admin_email' => 'info@gavahito.ir',
    'site_name' => 'Gavahito Admin Panel',
    'site_url' => 'https://gavahito.ir/',
    'maintenance_mode' => '0',
    'maintenance_message' => 'سیستم در حال نگهداری است. لطفاً کمی صبر کنید.',
    'backup_enabled' => '0',
    'backup_frequency' => 'daily',
    'log_retention_days' => '30',
    'max_login_attempts' => '5',
    'lockout_duration_minutes' => '15',
    'password_min_length' => '8',
    'require_strong_password' => '1',
    'twofa_enabled' => '0',
    'twofa_code_expiry' => '300',
    'email_notifications' => '1',
    'auto_logout_minutes' => '30',
    'favicon_path' => ''
];

foreach ($defaults as $key => $defaultValue) {
    if (!isset($currentSettings[$key])) {
        $currentSettings[$key] = $defaultValue;
    }
}

// Get current favicon path
$currentFavicon = $currentSettings['favicon_path'] ?? '';
?>
<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>تنظیمات سیستم - پنل مدیریت گواهیتو</title>
    <link rel="stylesheet" href="assets/admin-style.css">
    <?php if ($currentFavicon && file_exists($currentFavicon)): ?>
        <link rel="icon" type="image/x-icon" href="<?php echo htmlspecialchars($currentFavicon); ?>">
    <?php else: ?>
        <link rel="icon" type="image/x-icon" href="data:image/svg+xml,<svg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 100 100'><text y='.9em' font-size='90'>⚙️</text></svg>">
    <?php endif; ?>
    <?php include 'includes/admin-styles.php'; ?>
</head>
<body>
    <?php include 'includes/header.php'; ?>
    <?php include 'includes/sidebar.php'; ?>
    
    <!-- Sidebar Overlay for Mobile -->
    <div class="sidebar-overlay" id="sidebarOverlay"></div>

    <!-- Main Content -->
    <div class="main-content" id="mainContent">
        <div class="page-header">
            <h1 class="page-title">⚙️ تنظیمات سیستم</h1>
            <p class="page-description">مدیریت تنظیمات و پیکربندی سیستم</p>
        </div>

        <?php if ($error): ?>
            <div class="alert alert-danger">
                <?php echo htmlspecialchars($error); ?>
            </div>
        <?php endif; ?>

        <?php if ($message): ?>
            <div class="alert alert-success">
                <?php echo htmlspecialchars($message); ?>
            </div>
        <?php endif; ?>

        <!-- Settings Actions -->
        <div class="card" style="margin-bottom: 2rem;">
            <div class="card-header">
                <h2 class="card-title">🔄 عملیات تنظیمات</h2>
            </div>
            
            <div style="display: flex; gap: 1rem; flex-wrap: wrap;">
                <form method="POST" action="" style="display: inline;">
                    <input type="hidden" name="action" value="export_settings">
                    <button type="submit" class="btn btn-primary">
                        📤 خروجی تنظیمات
                    </button>
                </form>
                
                <form method="POST" action="" enctype="multipart/form-data" style="display: inline;">
                    <input type="hidden" name="action" value="import_settings">
                    <input type="file" name="settings_file" accept=".json" style="display: none;" id="settingsFile" onchange="this.form.submit()">
                    <button type="button" class="btn btn-secondary" onclick="document.getElementById('settingsFile').click()">
                        📥 ورودی تنظیمات
                    </button>
                </form>
                
                <form method="POST" action="" style="display: inline;">
                    <input type="hidden" name="action" value="reset_settings">
                    <button type="submit" class="btn btn-warning" onclick="return confirm('آیا از بازگردانی تنظیمات به حالت پیش‌فرض اطمینان دارید؟')">
                        🔄 بازگردانی پیش‌فرض
                    </button>
                </form>
            </div>
        </div>

        <!-- Settings Form -->
        <form method="POST" action="" enctype="multipart/form-data">
            <input type="hidden" name="action" value="update_settings">
            
            <!-- General Settings -->
            <div class="card">
                <div class="card-header">
                    <h2 class="card-title">🌐 تنظیمات عمومی</h2>
                </div>
                
                <div class="grid grid-cols-1 grid-cols-2" style="gap: 1rem;">
                    <div class="form-group">
                        <label for="site_name" class="form-label">نام سایت</label>
                        <input type="text" id="site_name" name="site_name" class="form-input" 
                               value="<?php echo htmlspecialchars($currentSettings['site_name']); ?>" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="site_url" class="form-label">آدرس سایت</label>
                        <input type="url" id="site_url" name="site_url" class="form-input" 
                               value="<?php echo htmlspecialchars($currentSettings['site_url']); ?>" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="admin_email" class="form-label">ایمیل مدیر</label>
                        <input type="email" id="admin_email" name="admin_email" class="form-input" 
                               value="<?php echo htmlspecialchars($currentSettings['admin_email']); ?>" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="maintenance_mode" class="form-label">حالت نگهداری</label>
                        <select id="maintenance_mode" name="maintenance_mode" class="form-select">
                            <option value="0" <?php echo $currentSettings['maintenance_mode'] == '0' ? 'selected' : ''; ?>>غیرفعال</option>
                            <option value="1" <?php echo $currentSettings['maintenance_mode'] == '1' ? 'selected' : ''; ?>>فعال</option>
                        </select>
                    </div>
                    
                    <div class="form-group" style="grid-column: span 2;">
                        <label for="maintenance_message" class="form-label">پیام نگهداری</label>
                        <textarea id="maintenance_message" name="maintenance_message" class="form-textarea" rows="3"><?php echo htmlspecialchars($currentSettings['maintenance_message']); ?></textarea>
                    </div>
                    
                    <!-- Favicon Upload Section -->
                    <div class="form-group" style="grid-column: span 2;">
                        <label class="form-label">فاوآیکون سایت</label>
                        <div style="display: flex; gap: 1rem; align-items: center; flex-wrap: wrap;">
                            <?php if ($currentFavicon && file_exists($currentFavicon)): ?>
                                <div style="display: flex; align-items: center; gap: 0.5rem;">
                                    <img src="<?php echo htmlspecialchars($currentFavicon); ?>" alt="فاوآیکون فعلی" style="width: 32px; height: 32px; border-radius: 4px;">
                                    <span style="color: var(--dark-300); font-size: 0.9rem;">فاوآیکون فعلی</span>
                                </div>
                            <?php endif; ?>
                            
                            <input type="file" name="favicon" accept=".ico,.png,.jpg,.jpeg,.gif" style="display: none;" id="faviconFile">
                            <button type="button" class="btn btn-info" onclick="document.getElementById('faviconFile').click()">
                                📁 انتخاب فاوآیکون
                            </button>
                            
                            <div style="color: var(--dark-400); font-size: 0.85rem;">
                                فرمت‌های مجاز: ICO، PNG، JPEG، GIF (حداکثر 1MB)
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Security Settings -->
            <div class="card">
                <div class="card-header">
                    <h2 class="card-title">🔒 تنظیمات امنیتی</h2>
                </div>
                
                <div class="grid grid-cols-1 grid-cols-2" style="gap: 1rem;">
                    <div class="form-group">
                        <label for="rate_limit_per_hour" class="form-label">محدودیت درخواست در ساعت</label>
                        <input type="number" id="rate_limit_per_hour" name="rate_limit_per_hour" class="form-input" 
                               value="<?php echo htmlspecialchars($currentSettings['rate_limit_per_hour']); ?>" min="1" max="100">
                    </div>
                    
                    <div class="form-group">
                        <label for="session_timeout_minutes" class="form-label">مدت زمان جلسه (دقیقه)</label>
                        <input type="number" id="session_timeout_minutes" name="session_timeout_minutes" class="form-input" 
                               value="<?php echo htmlspecialchars($currentSettings['session_timeout_minutes']); ?>" min="5" max="480">
                    </div>
                    
                    <div class="form-group">
                        <label for="max_login_attempts" class="form-label">حداکثر تلاش ورود</label>
                        <input type="number" id="max_login_attempts" name="max_login_attempts" class="form-input" 
                               value="<?php echo htmlspecialchars($currentSettings['max_login_attempts']); ?>" min="1" max="20">
                    </div>
                    
                    <div class="form-group">
                        <label for="lockout_duration_minutes" class="form-label">مدت زمان قفل (دقیقه)</label>
                        <input type="number" id="lockout_duration_minutes" name="lockout_duration_minutes" class="form-input" 
                               value="<?php echo htmlspecialchars($currentSettings['lockout_duration_minutes']); ?>" min="1" max="1440">
                    </div>
                    
                    <div class="form-group">
                        <label for="password_min_length" class="form-label">حداقل طول رمز عبور</label>
                        <input type="number" id="password_min_length" name="password_min_length" class="form-input" 
                               value="<?php echo htmlspecialchars($currentSettings['password_min_length']); ?>" min="6" max="50">
                    </div>
                    
                    <div class="form-group">
                        <label for="auto_logout_minutes" class="form-label">خروج خودکار (دقیقه)</label>
                        <input type="number" id="auto_logout_minutes" name="auto_logout_minutes" class="form-input" 
                               value="<?php echo htmlspecialchars($currentSettings['auto_logout_minutes']); ?>" min="5" max="480">
                    </div>
                    
                    <div class="form-group">
                        <label for="enable_xss_protection" class="form-label">محافظت XSS</label>
                        <select id="enable_xss_protection" name="enable_xss_protection" class="form-select">
                            <option value="1" <?php echo $currentSettings['enable_xss_protection'] == '1' ? 'selected' : ''; ?>>فعال</option>
                            <option value="0" <?php echo $currentSettings['enable_xss_protection'] == '0' ? 'selected' : ''; ?>>غیرفعال</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label for="enable_sql_injection_protection" class="form-label">محافظت SQL Injection</label>
                        <select id="enable_sql_injection_protection" name="enable_sql_injection_protection" class="form-select">
                            <option value="1" <?php echo $currentSettings['enable_sql_injection_protection'] == '1' ? 'selected' : ''; ?>>فعال</option>
                            <option value="0" <?php echo $currentSettings['enable_sql_injection_protection'] == '0' ? 'selected' : ''; ?>>غیرفعال</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label for="require_strong_password" class="form-label">رمز عبور قوی</label>
                        <select id="require_strong_password" name="require_strong_password" class="form-select">
                            <option value="1" <?php echo $currentSettings['require_strong_password'] == '1' ? 'selected' : ''; ?>>فعال</option>
                            <option value="0" <?php echo $currentSettings['require_strong_password'] == '0' ? 'selected' : ''; ?>>غیرفعال</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label for="twofa_enabled" class="form-label">احراز هویت دو مرحله‌ای</label>
                        <select id="twofa_enabled" name="twofa_enabled" class="form-select">
                            <option value="0" <?php echo $currentSettings['twofa_enabled'] == '0' ? 'selected' : ''; ?>>غیرفعال</option>
                            <option value="1" <?php echo $currentSettings['twofa_enabled'] == '1' ? 'selected' : ''; ?>>فعال</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label for="twofa_code_expiry" class="form-label">مدت اعتبار کد تایید (ثانیه)</label>
                        <input type="number" id="twofa_code_expiry" name="twofa_code_expiry" class="form-input" 
                               value="<?php echo htmlspecialchars($currentSettings['twofa_code_expiry'] ?? '300'); ?>" min="60" max="3600">
                    </div>
                    
                    <div class="form-group">
                        <label for="enable_recaptcha" class="form-label">reCAPTCHA</label>
                        <select id="enable_recaptcha" name="enable_recaptcha" class="form-select">
                            <option value="1" <?php echo $currentSettings['enable_recaptcha'] == '1' ? 'selected' : ''; ?>>فعال</option>
                            <option value="0" <?php echo $currentSettings['enable_recaptcha'] == '0' ? 'selected' : ''; ?>>غیرفعال</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label for="email_notifications" class="form-label">اعلان‌های ایمیل</label>
                        <select id="email_notifications" name="email_notifications" class="form-select">
                            <option value="1" <?php echo $currentSettings['email_notifications'] == '1' ? 'selected' : ''; ?>>فعال</option>
                            <option value="0" <?php echo $currentSettings['email_notifications'] == '0' ? 'selected' : ''; ?>>غیرفعال</option>
                        </select>
                    </div>
                </div>
            </div>

            <!-- System Settings -->
            <div class="card">
                <div class="card-header">
                    <h2 class="card-title">💾 تنظیمات سیستم</h2>
                </div>
                
                <div class="grid grid-cols-1 grid-cols-2" style="gap: 1rem;">
                    <div class="form-group">
                        <label for="backup_enabled" class="form-label">پشتیبان‌گیری خودکار</label>
                        <select id="backup_enabled" name="backup_enabled" class="form-select">
                            <option value="0" <?php echo $currentSettings['backup_enabled'] == '0' ? 'selected' : ''; ?>>غیرفعال</option>
                            <option value="1" <?php echo $currentSettings['backup_enabled'] == '1' ? 'selected' : ''; ?>>فعال</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label for="backup_frequency" class="form-label">فرکانس پشتیبان‌گیری</label>
                        <select id="backup_frequency" name="backup_frequency" class="form-select">
                            <option value="daily" <?php echo $currentSettings['backup_frequency'] == 'daily' ? 'selected' : ''; ?>>روزانه</option>
                            <option value="weekly" <?php echo $currentSettings['backup_frequency'] == 'weekly' ? 'selected' : ''; ?>>هفتگی</option>
                            <option value="monthly" <?php echo $currentSettings['backup_frequency'] == 'monthly' ? 'selected' : ''; ?>>ماهانه</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label for="log_retention_days" class="form-label">نگهداری لاگ (روز)</label>
                        <input type="number" id="log_retention_days" name="log_retention_days" class="form-input" 
                               value="<?php echo htmlspecialchars($currentSettings['log_retention_days']); ?>" min="1" max="365">
                    </div>
                </div>
                

            </div>

            <!-- Save Button -->
            <div class="card">
                <div style="text-align: center;">
                    <input type="hidden" name="action" value="update_settings">
                    <button type="submit" class="btn btn-primary btn-lg">
                        💾 ذخیره تنظیمات
                    </button>
                </div>
            </div>
        </form>
    </div>

    <script>
        // Form validation
        document.querySelector('form').addEventListener('submit', function(e) {
            const siteUrl = document.getElementById('site_url').value;
            const adminEmail = document.getElementById('admin_email').value;
            
            // Validate URL
            if (siteUrl && !isValidURL(siteUrl)) {
                e.preventDefault();
                alert('لطفاً آدرس سایت معتبر وارد کنید');
                document.getElementById('site_url').focus();
                return false;
            }
            
            // Validate email
            if (adminEmail && !isValidEmail(adminEmail)) {
                e.preventDefault();
                alert('لطفاً ایمیل معتبر وارد کنید');
                document.getElementById('admin_email').focus();
                return false;
            }
            
            // Validate numeric fields
            const numericFields = ['rate_limit_per_hour', 'session_timeout_minutes', 'max_login_attempts', 
                                 'lockout_duration_minutes', 'password_min_length', 'auto_logout_minutes', 'log_retention_days', 'twofa_code_expiry'];
            
            numericFields.forEach(fieldId => {
                const field = document.getElementById(fieldId);
                if (field && field.value && isNaN(field.value)) {
                    e.preventDefault();
                    alert('لطفاً مقدار عددی معتبر وارد کنید');
                    field.focus();
                    return false;
                }
            });
        });
        
        // URL validation function
        function isValidURL(string) {
            try {
                new URL(string);
                return true;
            } catch (_) {
                return false;
            }
        }
        
        // Email validation function
        function isValidEmail(email) {
            const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            return emailRegex.test(email);
        }
        
        // Auto-save settings every 5 minutes
        let autoSaveTimer;
        function startAutoSave() {
            autoSaveTimer = setTimeout(function() {
                if (confirm('آیا می‌خواهید تنظیمات به صورت خودکار ذخیره شود؟')) {
                    document.querySelector('form').submit();
                } else {
                    startAutoSave(); // Restart timer
                }
            }, 300000); // 5 minutes
        }
        
        // Start auto-save when page loads
        document.addEventListener('DOMContentLoaded', function() {
            startAutoSave();
        });
        
        // Reset timer when form is modified
        document.querySelectorAll('input, select, textarea').forEach(element => {
            element.addEventListener('change', function() {
                clearTimeout(autoSaveTimer);
                startAutoSave();
            });
        });
        
        // File upload preview
        document.getElementById('faviconFile')?.addEventListener('change', function(e) {
            const file = e.target.files[0];
            if (file) {
                // Show loading message
                const button = this.parentElement.querySelector('button');
                const originalText = button.innerHTML;
                button.innerHTML = '⏳ در حال آپلود...';
                button.disabled = true;
                
                // Show file info
                const fileInfo = document.createElement('div');
                fileInfo.style.color = 'var(--success-500)';
                fileInfo.style.fontSize = '0.9rem';
                fileInfo.style.marginTop = '0.5rem';
                fileInfo.innerHTML = `📁 فایل انتخاب شده: ${file.name} (${(file.size / 1024).toFixed(1)} KB)`;
                this.parentElement.appendChild(fileInfo);
                
                // Submit form after a short delay to show loading state
                setTimeout(() => {
                    this.form.submit();
                }, 100);
            }
        });
    </script>
</body>
</html>
