<?php
// ========================================
// SECURITY HEADERS AND CONFIGURATIONS
// ========================================

// Prevent direct access
if (!defined('SECURE_ACCESS')) {
    define('SECURE_ACCESS', true);
}

// Security headers
header('X-Content-Type-Options: nosniff');
header('X-Frame-Options: DENY');
header('X-XSS-Protection: 1; mode=block');
header('Referrer-Policy: strict-origin-when-cross-origin');
header('Content-Security-Policy: default-src \'self\'; script-src \'self\' \'unsafe-inline\'; style-src \'self\' \'unsafe-inline\'; img-src \'self\' data:; font-src \'self\'; connect-src \'self\';');
header('Strict-Transport-Security: max-age=31536000; includeSubDomains; preload');
header('Permissions-Policy: geolocation=(), microphone=(), camera=()');

// Set proper charset headers
header('Content-Type: text/html; charset=utf-8');
mb_internal_encoding('UTF-8');
mb_http_output('UTF-8');

// Enable error reporting for development
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Start session with basic settings for development
if (session_status() === PHP_SESSION_NONE) {
    // Basic session configuration for development
    ini_set('session.cookie_httponly', 1);
    ini_set('session.use_strict_mode', 1);
    ini_set('session.gc_maxlifetime', 3600); // 1 hour
    ini_set('session.cookie_lifetime', 0); // Session cookie
    
    session_start();
}

require_once 'config/database.php';
require_once 'includes/functions.php';
require_once 'config/email-config.php';

// ========================================
// ADVANCED SECURITY FUNCTIONS
// ========================================

/**
 * Advanced input validation and sanitization
 */
function advancedSanitizeInput($input, $type = 'string', $maxLength = 255) {
    if (is_array($input)) {
        return array_map(function($item) use ($type, $maxLength) {
            return advancedSanitizeInput($item, $type, $maxLength);
        }, $input);
    }
    
    $input = trim($input);
    
    // Length validation
    if (strlen($input) > $maxLength) {
        return false;
    }
    
    switch ($type) {
        case 'username':
            // Username: alphanumeric, underscore, dash, 3-20 chars
            if (!preg_match('/^[a-zA-Z0-9_-]{3,20}$/', $input)) {
                return false;
            }
            return htmlspecialchars($input, ENT_QUOTES, 'UTF-8');
            
        case 'email':
            // Email validation
            if (!filter_var($input, FILTER_VALIDATE_EMAIL)) {
                return false;
            }
            return filter_var($input, FILTER_SANITIZE_EMAIL);
            
        case 'password':
            // Password: minimum 8 chars, at least one letter and one number
            if (strlen($input) < 8 || !preg_match('/^(?=.*[A-Za-z])(?=.*\d)[A-Za-z\d@$!%*?&]{8,}$/', $input)) {
                return false;
            }
            return $input; // Don't sanitize passwords
            
        case 'integer':
            return filter_var($input, FILTER_VALIDATE_INT) !== false ? (int)$input : false;
            
        case 'verification_code':
            // 6-digit numeric code
            if (!preg_match('/^[0-9]{6}$/', $input)) {
                return false;
            }
            return $input;
            
        default:
            return htmlspecialchars($input, ENT_QUOTES, 'UTF-8');
    }
}



/**
 * CSRF protection
 */
function generateCSRFToken() {
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

function validateCSRFToken($token) {
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}

/**
 * Advanced logging for security events (using existing function from functions.php)
 */
function logSecurityEventAdvanced($event, $details, $severity = 'medium', $user = null) {
    $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
    $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';
    $user = $user ?? ($_SESSION['admin_id'] ?? 'unknown');
    
    $logEntry = [
        'timestamp' => date('Y-m-d H:i:s'),
        'event' => $event,
        'details' => $details,
        'severity' => $severity,
        'user' => $user,
        'ip' => $ip,
        'user_agent' => $userAgent,
        'session_id' => session_id(),
        'request_uri' => $_SERVER['REQUEST_URI'] ?? '',
        'request_method' => $_SERVER['REQUEST_METHOD'] ?? ''
    ];
    
    $logFile = __DIR__ . '/logs/security.log';
    $logDir = dirname($logFile);
    
    if (!is_dir($logDir)) {
        mkdir($logDir, 0755, true);
    }
    
    file_put_contents($logFile, json_encode($logEntry) . "\n", FILE_APPEND | LOCK_EX);
}

/**
 * Validate and sanitize operation type
 */
function validateOperationType($operation) {
    $allowedOperations = [
        'delete', 'enable_protection', 'disable_protection', 
        'activate', 'deactivate', 'send_create_verification', 
        'create_user', 'send_verification_code', 'execute_bulk_operation',
        'update', 'change_password'
    ];
    
    return in_array($operation, $allowedOperations, true);
}

// Check authentication
if (!isLoggedIn()) {
    logSecurityEvent('unauthorized_access_attempt', 'Unauthorized access attempt to users.php from IP: ' . getClientIP(), 'high');
    header('Location: 403.php');
    exit;
}
checkSessionTimeout();

$message = '';
$error = '';

// Handle user actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // CSRF protection disabled for development
    // if (!isset($_POST['csrf_token']) || !validateCSRFToken($_POST['csrf_token'])) {
    //     logSecurityEvent('csrf_attack_detected', 'Invalid or missing CSRF token', 'high');
    //     http_response_code(403);
    //     die('Access denied: Invalid CSRF token');
    // }
    
    $action = $_POST['action'] ?? '';
    
    // Validate operation type
    if (!validateOperationType($action)) {
        logSecurityEventAdvanced('invalid_operation_attempt', "Invalid operation: $action", 'high');
        http_response_code(400);
        die('Invalid operation');
    }
    
    // Rate limiting disabled for development
    // if (!checkRateLimit($action, 10)) { // 10 attempts per hour (using functions.php)
    //     logSecurityEvent('rate_limit_exceeded', "Action: $action", 'high');
    //     http_response_code(429);
    //     die('Too many requests. Please try again later.');
    // }
    
    try {
        $db = new Database();
        $conn = $db->getConnection();
        
        switch ($action) {
            case 'send_create_verification':
                // Advanced input validation
                $username = advancedSanitizeInput($_POST['username'] ?? '', 'username', 20);
                $password = advancedSanitizeInput($_POST['password'] ?? '', 'password', 128);
                $email = advancedSanitizeInput($_POST['email'] ?? '', 'email', 255);
                $adminEmail = 'raytheme.ir@gmail.com'; // ایمیل مدیریت ثابت
                
                // Validate inputs
                if ($username === false || $password === false || ($email !== '' && $email === false)) {
                    logSecurityEventAdvanced('invalid_input_attempt', 'Invalid input in user creation', 'medium');
                    $error = 'اطلاعات وارد شده نامعتبر است';
                    // Return error response for AJAX requests
                    if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && $_SERVER['HTTP_X_REQUESTED_WITH'] === 'XMLHttpRequest') {
                        header('Content-Type: application/json');
                        echo json_encode([
                            'success' => false,
                            'error' => $error
                        ]);
                        exit;
                    }
                } elseif (empty($username) || empty($password)) {
                    $error = 'اطلاعات ناقص است';
                    // Return error response for AJAX requests
                    if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && $_SERVER['HTTP_X_REQUESTED_WITH'] === 'XMLHttpRequest') {
                        header('Content-Type: application/json');
                        echo json_encode([
                            'success' => false,
                            'error' => $error
                        ]);
                        exit;
                    }
                } else {
                    // Check if username exists
                    $checkQuery = "SELECT id FROM admin_users WHERE username = ?";
                    $checkStmt = $conn->prepare($checkQuery);
                    $checkStmt->execute([$username]);
                    
                    if ($checkStmt->fetch()) {
                        $error = 'این نام کاربری قبلاً وجود دارد';
                        // Return error response for AJAX requests
                        if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && $_SERVER['HTTP_X_REQUESTED_WITH'] === 'XMLHttpRequest') {
                            header('Content-Type: application/json');
                            echo json_encode([
                                'success' => false,
                                'error' => $error
                            ]);
                            exit;
                        }
                    } else {
                        // Generate verification code
                        $verificationCode = sprintf('%06d', mt_rand(100000, 999999));
                        
                        // Store user data in session
                        $_SESSION['pending_user_creation'] = [
                            'username' => $username,
                            'password' => $password,
                            'email' => $email,
                            'code' => $verificationCode,
                            'expires' => time() + 300 // 5 minutes
                        ];
                        
                        // Send verification email using advanced email system
                        $subject = '🔐 کد تأیید ایجاد کاربر جدید - ' . SITE_NAME;
                        $message_body = '
                        <h3 style="color: #667eea; margin-bottom: 20px;">🔐 کد تأیید ایجاد کاربر جدید</h3>
                        
                        <div class="info-row">
                            <span class="info-label">👤 نام کاربری:</span>
                            <span class="info-value">' . htmlspecialchars($username) . '</span>
                        </div>
                        
                        <div class="info-row">
                            <span class="info-label">📧 ایمیل کاربر:</span>
                            <span class="info-value">' . htmlspecialchars($email ?: 'تنظیم نشده') . '</span>
                        </div>
                        
                        <div style="margin-top: 20px;">
                            <div class="info-label" style="margin-bottom: 10px;">🔢 کد تأیید:</div>
                            <div class="description-box" style="text-align: center; font-size: 24px; font-weight: bold; letter-spacing: 8px; color: #667eea;">
                                ' . $verificationCode . '
                            </div>
                        </div>
                        
                        <div style="margin-top: 20px;">
                            <div class="description-box" style="background: #fff3cd; border-color: #ffeaa7; color: #856404;">
                                <strong>⚠️ توجه:</strong> این کد تا 5 دقیقه معتبر است. اگر شما این درخواست را نکرده‌اید، این ایمیل را نادیده بگیرید.
                            </div>
                        </div>';
                        
                        if (sendEmail($adminEmail, $subject, $message_body)) {
                            $message = "کد تأیید به ایمیل شما ارسال شد";
                            // Return JSON response for AJAX requests
                            if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && $_SERVER['HTTP_X_REQUESTED_WITH'] === 'XMLHttpRequest') {
                                header('Content-Type: application/json');
                                echo json_encode([
                                    'success' => true,
                                    'message' => $message
                                ]);
                                exit;
                            }
                        } else {
                            $error = 'خطا در ارسال کد تأیید';
                            // Return error response for AJAX requests
                            if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && $_SERVER['HTTP_X_REQUESTED_WITH'] === 'XMLHttpRequest') {
                                header('Content-Type: application/json');
                                echo json_encode([
                                    'success' => false,
                                    'error' => $error
                                ]);
                                exit;
                            }
                        }
                    }
                }
                break;
                
            case 'create_user':
                $verificationCode = advancedSanitizeInput($_POST['verification_code'] ?? '', 'verification_code', 6);
                
                if ($verificationCode === false) {
                    logSecurityEvent('invalid_verification_code', 'Invalid verification code format', 'medium');
                    $error = 'کد تأیید نامعتبر است';
                    // Return error response for AJAX requests
                    if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && $_SERVER['HTTP_X_REQUESTED_WITH'] === 'XMLHttpRequest') {
                        header('Content-Type: application/json');
                        echo json_encode([
                            'success' => false,
                            'error' => $error
                        ]);
                        exit;
                    }
                } elseif (!isset($_SESSION['pending_user_creation']) || 
                    $_SESSION['pending_user_creation']['code'] !== $verificationCode ||
                    $_SESSION['pending_user_creation']['expires'] < time()) {
                    logSecurityEvent('invalid_verification_attempt', 'Invalid or expired verification code', 'medium');
                    $error = 'کد تأیید نامعتبر یا منقضی شده است';
                    // Return error response for AJAX requests
                    if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && $_SERVER['HTTP_X_REQUESTED_WITH'] === 'XMLHttpRequest') {
                        header('Content-Type: application/json');
                        echo json_encode([
                            'success' => false,
                            'error' => $error
                        ]);
                        exit;
                    }
                } else {
                    $userData = $_SESSION['pending_user_creation'];
                    
                    // Create strong password hash using Argon2id
                    $passwordHash = password_hash($userData['password'], PASSWORD_ARGON2ID, [
                        'memory_cost' => 131072,
                        'time_cost' => 8,
                        'threads' => 4
                    ]);
                    
                    $insertQuery = "INSERT INTO admin_users (username, password, email, is_active, is_protected, created_at) VALUES (?, ?, ?, 1, 0, NOW())";
                    $insertStmt = $conn->prepare($insertQuery);
                    $insertStmt->execute([$userData['username'], $passwordHash, $userData['email']]);
                    
                    $message = "کاربر جدید با موفقیت ایجاد شد";
                    logSecurityEvent('user_created', 'New user: ' . $userData['username'], 'low');
                    
                    // Clear pending creation
                    unset($_SESSION['pending_user_creation']);
                    
                    // Return JSON response for AJAX requests
                    if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && $_SERVER['HTTP_X_REQUESTED_WITH'] === 'XMLHttpRequest') {
                        header('Content-Type: application/json');
                        echo json_encode([
                            'success' => true,
                            'message' => $message
                        ]);
                        exit;
                    }
                }
                break;
                
            case 'send_verification_code':
                $operation = advancedSanitizeInput($_POST['operation'] ?? '', 'string', 50);
                $selectedUsers = advancedSanitizeInput($_POST['selected_users'] ?? [], 'integer', 10);
                $adminEmail = 'raytheme.ir@gmail.com'; // ایمیل مدیریت ثابت
                
                // Validate operation type
                if (!validateOperationType($operation)) {
                    logSecurityEvent('invalid_operation_attempt', "Invalid operation: $operation", 'high');
                    $error = 'عملیات نامعتبر است';
                    // Return error response for AJAX requests
                    if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && $_SERVER['HTTP_X_REQUESTED_WITH'] === 'XMLHttpRequest') {
                        header('Content-Type: application/json');
                        echo json_encode([
                            'success' => false,
                            'error' => $error
                        ]);
                        exit;
                    }
                } elseif (empty($selectedUsers) || empty($operation)) {
                    $error = 'اطلاعات ناقص است';
                    // Return error response for AJAX requests
                    if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && $_SERVER['HTTP_X_REQUESTED_WITH'] === 'XMLHttpRequest') {
                        header('Content-Type: application/json');
                        echo json_encode([
                            'success' => false,
                            'error' => $error
                        ]);
                        exit;
                    }
                } else {
                    // Generate unique operation ID
                    $operationId = uniqid('op_', true);
                    $verificationCode = sprintf('%06d', mt_rand(100000, 999999));
                    
                    // Store operation details in session
                    $_SESSION['pending_operation'] = [
                        'id' => $operationId,
                        'type' => $operation,
                        'users' => $selectedUsers,
                        'code' => $verificationCode,
                        'expires' => time() + 300 // 5 minutes
                    ];
                    
                    // Get operation details for email
                    $operationDetails = [
                        'delete' => ['title' => 'حذف کاربران', 'icon' => '🗑️', 'color' => '#e74c3c'],
                        'enable_protection' => ['title' => 'فعال‌سازی محافظ', 'icon' => '🛡️', 'color' => '#27ae60'],
                        'disable_protection' => ['title' => 'غیرفعال‌سازی محافظ', 'icon' => '🔓', 'color' => '#f39c12'],
                        'activate' => ['title' => 'فعال‌سازی کاربران', 'icon' => '✅', 'color' => '#3498db'],
                        'deactivate' => ['title' => 'غیرفعال‌سازی کاربران', 'icon' => '❌', 'color' => '#9b59b6']
                    ];
                    
                    $opInfo = $operationDetails[$operation] ?? ['title' => 'عملیات نامشخص', 'icon' => '⚙️', 'color' => '#95a5a6'];
                    
                    // Send verification email using advanced email system
                    $subject = $opInfo['icon'] . ' کد تأیید عملیات مدیریت کاربران - ' . SITE_NAME;
                    $message_body = '
                    <h3 style="color: ' . $opInfo['color'] . '; margin-bottom: 20px;">' . $opInfo['icon'] . ' کد تأیید عملیات مدیریت کاربران</h3>
                    
                    <div class="info-row">
                        <span class="info-label">🔧 نوع عملیات:</span>
                        <span class="info-value">' . $opInfo['title'] . '</span>
                    </div>
                    
                    <div class="info-row">
                        <span class="info-label">👥 تعداد کاربران:</span>
                        <span class="info-value">' . count($selectedUsers) . ' کاربر</span>
                    </div>
                    
                    <div class="info-row">
                        <span class="info-label">🆔 شناسه عملیات:</span>
                        <span class="info-value">' . $operationId . '</span>
                    </div>
                    
                    <div style="margin-top: 20px;">
                        <div class="info-label" style="margin-bottom: 10px;">🔢 کد تأیید:</div>
                        <div class="description-box" style="text-align: center; font-size: 24px; font-weight: bold; letter-spacing: 8px; color: ' . $opInfo['color'] . ';">
                            ' . $verificationCode . '
                        </div>
                    </div>
                    
                    <div style="margin-top: 20px;">
                        <div class="description-box" style="background: #fff3cd; border-color: #ffeaa7; color: #856404;">
                            <strong>⚠️ توجه:</strong> این کد تا 5 دقیقه معتبر است. اگر شما این درخواست را نکرده‌اید، این ایمیل را نادیده بگیرید.
                        </div>
                    </div>';
                    
                    if (sendEmail($adminEmail, $subject, $message_body)) {
                        $message = "کد تأیید به ایمیل شما ارسال شد";
                        // Return operation ID in JSON response for AJAX requests
                        if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && $_SERVER['HTTP_X_REQUESTED_WITH'] === 'XMLHttpRequest') {
                            header('Content-Type: application/json');
                            echo json_encode([
                                'success' => true,
                                'message' => $message,
                                'operation_id' => $operationId
                            ]);
                            exit;
                        }
                    } else {
                        $error = 'خطا در ارسال کد تأیید';
                        // Return error response for AJAX requests
                        if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && $_SERVER['HTTP_X_REQUESTED_WITH'] === 'XMLHttpRequest') {
                            header('Content-Type: application/json');
                            echo json_encode([
                                'success' => false,
                                'error' => $error
                            ]);
                            exit;
                        }
                    }
                }
                break;
                
            case 'execute_bulk_operation':
                $operation = advancedSanitizeInput($_POST['operation'] ?? '', 'string', 50);
                $verificationCode = advancedSanitizeInput($_POST['verification_code'] ?? '', 'verification_code', 6);
                $operationId = advancedSanitizeInput($_POST['operation_id'] ?? '', 'string', 100);
                
                // Validate operation type
                if (!validateOperationType($operation)) {
                    logSecurityEvent('invalid_operation_attempt', "Invalid operation: $operation", 'high');
                    $error = 'عملیات نامعتبر است';
                    // Return error response for AJAX requests
                    if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && $_SERVER['HTTP_X_REQUESTED_WITH'] === 'XMLHttpRequest') {
                        header('Content-Type: application/json');
                        echo json_encode([
                            'success' => false,
                            'error' => $error
                        ]);
                        exit;
                    }
                } elseif ($verificationCode === false) {
                    logSecurityEvent('invalid_verification_code', 'Invalid verification code format', 'medium');
                    $error = 'کد تأیید نامعتبر است';
                    // Return error response for AJAX requests
                    if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && $_SERVER['HTTP_X_REQUESTED_WITH'] === 'XMLHttpRequest') {
                        header('Content-Type: application/json');
                        echo json_encode([
                            'success' => false,
                            'error' => $error
                        ]);
                        exit;
                    }
                } elseif (!isset($_SESSION['pending_operation']) || 
                    $_SESSION['pending_operation']['id'] !== $operationId ||
                    $_SESSION['pending_operation']['code'] !== $verificationCode ||
                    $_SESSION['pending_operation']['expires'] < time()) {
                    logSecurityEvent('invalid_verification_attempt', 'Invalid or expired verification code', 'medium');
                    $error = 'کد تأیید نامعتبر یا منقضی شده است';
                    // Return error response for AJAX requests
                    if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && $_SERVER['HTTP_X_REQUESTED_WITH'] === 'XMLHttpRequest') {
                        header('Content-Type: application/json');
                        echo json_encode([
                            'success' => false,
                            'error' => $error
                        ]);
                        exit;
                    }
                } else {
                    $selectedUsers = $_SESSION['pending_operation']['users'];
                    
                    switch ($operation) {
                        case 'delete':
                            // Filter out current user and protected users
                            $filteredUsers = array_filter($selectedUsers, function($userId) use ($conn) {
                                if ($userId == $_SESSION['admin_id']) return false;
                                
                                $checkQuery = "SELECT is_protected FROM admin_users WHERE id = ?";
                                $checkStmt = $conn->prepare($checkQuery);
                                $checkStmt->execute([$userId]);
                                $user = $checkStmt->fetch();
                                
                                return !$user || !$user['is_protected'];
                            });
                            
                            if (empty($filteredUsers)) {
                                $error = 'نمی‌توانید حساب کاربری خود یا کاربران محافظت شده را حذف کنید';
                                // Return error response for AJAX requests
                                if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && $_SERVER['HTTP_X_REQUESTED_WITH'] === 'XMLHttpRequest') {
                                    header('Content-Type: application/json');
                                    echo json_encode([
                                        'success' => false,
                                        'error' => $error
                                    ]);
                                    exit;
                                }
                            } else {
                                if (count($filteredUsers) > 0) {
                                    $placeholders = str_repeat('?,', count($filteredUsers) - 1) . '?';
                                    $deleteQuery = "DELETE FROM admin_users WHERE id IN ($placeholders)";
                                    $deleteStmt = $conn->prepare($deleteQuery);
                                    $deleteStmt->execute($filteredUsers);
                                    
                                    $message = count($filteredUsers) . " کاربر با موفقیت حذف شد";
                                    logSecurityEvent('users_bulk_deleted', 'Count: ' . count($filteredUsers), 'medium');
                                } else {
                                    $error = 'هیچ کاربری برای حذف انتخاب نشده است';
                                }
                            }
                            break;
                            
                        case 'enable_protection':
                            if (count($selectedUsers) > 0) {
                                $placeholders = str_repeat('?,', count($selectedUsers) - 1) . '?';
                                $updateQuery = "UPDATE admin_users SET is_protected = 1 WHERE id IN ($placeholders)";
                                $updateStmt = $conn->prepare($updateQuery);
                                $updateStmt->execute($selectedUsers);
                                
                                $message = count($selectedUsers) . " کاربر محافظت شد";
                                logSecurityEvent('users_protection_enabled', 'Count: ' . count($selectedUsers), 'medium');
                            } else {
                                $error = 'هیچ کاربری انتخاب نشده است';
                            }
                            break;
                            
                        case 'disable_protection':
                            if (count($selectedUsers) > 0) {
                                $placeholders = str_repeat('?,', count($selectedUsers) - 1) . '?';
                                $updateQuery = "UPDATE admin_users SET is_protected = 0 WHERE id IN ($placeholders)";
                                $updateStmt = $conn->prepare($updateQuery);
                                $updateStmt->execute($selectedUsers);
                                
                                $message = count($selectedUsers) . " کاربر از محافظت خارج شد";
                                logSecurityEvent('users_protection_disabled', 'Count: ' . count($selectedUsers), 'medium');
                            } else {
                                $error = 'هیچ کاربری انتخاب نشده است';
                            }
                            break;
                            
                        case 'activate':
                            if (count($selectedUsers) > 0) {
                                $placeholders = str_repeat('?,', count($selectedUsers) - 1) . '?';
                                $updateQuery = "UPDATE admin_users SET is_active = 1 WHERE id IN ($placeholders)";
                                $updateStmt = $conn->prepare($updateQuery);
                                $updateStmt->execute($selectedUsers);
                                
                                $message = count($selectedUsers) . " کاربر فعال شد";
                                logSecurityEvent('users_activated', 'Count: ' . count($selectedUsers), 'medium');
                            } else {
                                $error = 'هیچ کاربری انتخاب نشده است';
                            }
                            break;
                            
                        case 'deactivate':
                            // Don't allow deactivating current user
                            $filteredUsers = array_filter($selectedUsers, function($userId) {
                                return $userId != $_SESSION['admin_id'];
                            });
                            
                            if (empty($filteredUsers)) {
                                $error = 'نمی‌توانید حساب کاربری خود را غیرفعال کنید';
                                // Return error response for AJAX requests
                                if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && $_SERVER['HTTP_X_REQUESTED_WITH'] === 'XMLHttpRequest') {
                                    header('Content-Type: application/json');
                                    echo json_encode([
                                        'success' => false,
                                        'error' => $error
                                    ]);
                                    exit;
                                }
                            } else {
                                if (count($filteredUsers) > 0) {
                                    $placeholders = str_repeat('?,', count($filteredUsers) - 1) . '?';
                                    $updateQuery = "UPDATE admin_users SET is_active = 0 WHERE id IN ($placeholders)";
                                    $updateStmt = $conn->prepare($updateQuery);
                                    $updateStmt->execute($filteredUsers);
                                    
                                    $message = count($filteredUsers) . " کاربر غیرفعال شد";
                                    logSecurityEvent('users_deactivated', 'Count: ' . count($filteredUsers), 'medium');
                                } else {
                                    $error = 'هیچ کاربری برای غیرفعال‌سازی انتخاب نشده است';
                                }
                            }
                            break;
                    }
                    
                    // Clear pending operation
                    unset($_SESSION['pending_operation']);
                    
                    // Return JSON response for AJAX requests
                    if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && $_SERVER['HTTP_X_REQUESTED_WITH'] === 'XMLHttpRequest') {
                        header('Content-Type: application/json');
                        echo json_encode([
                            'success' => true,
                            'message' => $message
                        ]);
                        exit;
                    }
                }
                break;
                
            case 'update':
                $userId = advancedSanitizeInput($_POST['user_id'] ?? 0, 'integer', 10);
                $email = advancedSanitizeInput($_POST['email'] ?? '', 'email', 255);
                
                // Validate inputs
                if ($userId === false || ($email !== '' && $email === false)) {
                    logSecurityEvent('invalid_input_attempt', 'Invalid input in user update', 'medium');
                    $error = 'اطلاعات وارد شده نامعتبر است';
                } elseif ($userId != $_SESSION['admin_id']) {
                    $error = 'شما فقط می‌توانید اطلاعات حساب کاربری خود را ویرایش کنید';
                    logSecurityEvent('unauthorized_user_edit_attempt', 'User ID: ' . $userId . ' by: ' . $_SESSION['admin_id'], 'high');
                } else {
                    // بررسی محافظت کاربر
                    $checkQuery = "SELECT is_protected FROM admin_users WHERE id = ?";
                    $checkStmt = $conn->prepare($checkQuery);
                    $checkStmt->execute([$userId]);
                    $user = $checkStmt->fetch();
                    
                    if ($user && $user['is_protected']) {
                        $error = 'کاربران محافظت شده نمی‌توانند اطلاعات خود را تغییر دهند';
                        logSecurityEvent('protected_user_edit_attempt', 'User ID: ' . $userId, 'medium');
                    } else {
                        $updateQuery = "UPDATE admin_users SET email = ? WHERE id = ?";
                        $updateStmt = $conn->prepare($updateQuery);
                        $updateStmt->execute([$email, $userId]);
                        
                        $message = "اطلاعات شما با موفقیت بروزرسانی شد";
                        logSecurityEvent('user_self_updated', 'User ID: ' . $userId, 'low');
                    }
                }
                break;
                
            case 'change_password':
                $userId = advancedSanitizeInput($_POST['user_id'] ?? 0, 'integer', 10);
                $newPassword = advancedSanitizeInput($_POST['new_password'] ?? '', 'password', 128);
                
                // Validate inputs
                if ($userId === false || $newPassword === false) {
                    logSecurityEvent('invalid_input_attempt', 'Invalid input in password change', 'medium');
                    $error = 'اطلاعات وارد شده نامعتبر است';
                } elseif ($userId != $_SESSION['admin_id']) {
                    $error = 'شما فقط می‌توانید رمز عبور خود را تغییر دهید';
                    logSecurityEvent('unauthorized_password_change_attempt', 'User ID: ' . $userId . ' by: ' . $_SESSION['admin_id'], 'high');
                } else {
                    // بررسی محافظت کاربر
                    $checkQuery = "SELECT is_protected FROM admin_users WHERE id = ?";
                    $checkStmt = $conn->prepare($checkQuery);
                    $checkStmt->execute([$userId]);
                    $user = $checkStmt->fetch();
                    
                    if ($user && $user['is_protected']) {
                        $error = 'کاربران محافظت شده نمی‌توانند رمز عبور خود را تغییر دهند';
                        logSecurityEvent('protected_user_password_change_attempt', 'User ID: ' . $userId, 'medium');
                    } elseif (empty($newPassword)) {
                        $error = 'رمز عبور جدید الزامی است';
                    } elseif (strlen($newPassword) < 8) {
                        $error = 'رمز عبور باید حداقل 8 کاراکتر باشد';
                    } else {
                        // Create strong password hash
                        $passwordHash = password_hash($newPassword, PASSWORD_ARGON2ID, [
                            'memory_cost' => 131072,
                            'time_cost' => 8,
                            'threads' => 4
                        ]);
                        
                        $updateQuery = "UPDATE admin_users SET password = ? WHERE id = ?";
                        $updateStmt = $conn->prepare($updateQuery);
                        $updateStmt->execute([$passwordHash, $userId]);
                        
                        $message = "رمز عبور شما با موفقیت تغییر یافت";
                        logSecurityEvent('user_self_password_changed', 'User ID: ' . $userId, 'medium');
                    }
                }
                break;
        }
    } catch (Exception $e) {
        error_log("Users page error: " . $e->getMessage());
        $error = 'خطا در عملیات. لطفاً دوباره تلاش کنید';
    }
}

// Get users list
try {
    $db = new Database();
    $conn = $db->getConnection();
    
    $usersQuery = "SELECT id, username, email, is_active, is_protected, last_login, created_at FROM admin_users ORDER BY created_at DESC";
    $usersStmt = $conn->query($usersQuery);
    $users = $usersStmt->fetchAll();
    
} catch (Exception $e) {
    error_log("Users list error: " . $e->getMessage());
    $error = 'خطا در بارگذاری لیست کاربران';
    $users = [];
}
?>
<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>مدیریت کاربران - پنل مدیریت گواهیتو</title>
    <link rel="stylesheet" href="assets/admin-style.css">
    <link rel="icon" type="image/png" href="assets/images/logo.png">
    <?php include 'includes/admin-styles.php'; ?>
    <style>
        /* Toast Notification Styles */
        .toast {
            position: fixed;
            top: 20px;
            right: 20px;
            padding: 1rem 1.5rem;
            border-radius: 0.5rem;
            color: white;
            font-weight: 500;
            z-index: 3000;
            transform: translateX(100%);
            transition: transform 0.3s ease;
            max-width: 400px;
            box-shadow: 0 10px 25px rgba(0, 0, 0, 0.2);
        }
        
        .toast.show {
            transform: translateX(0);
        }
        
        .toast-success {
            background: #059669;
            border-left: 4px solid #047857;
        }
        
        .toast-error {
            background: #dc2626;
            border-left: 4px solid #b91c1c;
        }
        
        .toast-warning {
            background: #d97706;
            border-left: 4px solid #b45309;
        }
        
        .bulk-actions {
            display: flex;
            gap: 0.5rem;
            flex-wrap: wrap;
            margin-bottom: 1rem;
        }
        
        .bulk-actions button {
            padding: 0.5rem 1rem;
            border: none;
            border-radius: 0.5rem;
            cursor: pointer;
            font-size: 0.875rem;
            font-family: inherit;
            font-weight: 500;
            transition: all 0.3s ease;
        }
        
        .bulk-actions button:disabled {
            opacity: 0.5;
            cursor: not-allowed;
        }
        
        .btn-delete { 
            background: #dc2626; 
            color: white; 
            font-weight: 500;
            text-shadow: 0 1px 2px rgba(0,0,0,0.1);
        }
        .btn-delete:hover { background: #b91c1c; }
        
        .btn-protect { 
            background: #059669; 
            color: white; 
            font-weight: 500;
            text-shadow: 0 1px 2px rgba(0,0,0,0.1);
        }
        .btn-protect:hover { background: #047857; }
        
        .btn-unprotect { 
            background: #d97706; 
            color: white; 
            font-weight: 500;
            text-shadow: 0 1px 2px rgba(0,0,0,0.1);
        }
        .btn-unprotect:hover { background: #b45309; }
        
        .btn-activate { 
            background: #2563eb; 
            color: white; 
            font-weight: 500;
            text-shadow: 0 1px 2px rgba(0,0,0,0.1);
        }
        .btn-activate:hover { background: #1d4ed8; }
        
        .btn-deactivate { 
            background: #7c3aed; 
            color: white; 
            font-weight: 500;
            text-shadow: 0 1px 2px rgba(0,0,0,0.1);
        }
        .btn-deactivate:hover { background: #6d28d9; }
        
        .users-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 1rem;
        }
        
        .users-table th,
        .users-table td {
            padding: 0.75rem;
            text-align: right;
            border-bottom: 1px solid var(--dark-600);
        }
        
        .users-table th {
            background: var(--dark-700);
            font-weight: 600;
            color: var(--primary-400);
        }
        
        .users-table tr:hover {
            background: var(--dark-700);
        }
        
        .user-checkbox {
            width: 18px;
            height: 18px;
            cursor: pointer;
        }
        
        .status-badge {
            padding: 0.25rem 0.5rem;
            border-radius: 0.25rem;
            font-size: 0.75rem;
            font-weight: 500;
        }
        
        .status-active { background: #059669; color: white; }
        .status-inactive { background: #dc2626; color: white; }
        .status-protected { background: #d97706; color: white; }
        
        /* Modal Styles */
        .modal {
            display: none;
            position: fixed;
            z-index: 2000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.5);
            backdrop-filter: blur(5px);
        }
        
        .modal-content {
            background: var(--dark-800);
            margin: 5% auto;
            padding: 2rem;
            border-radius: 1rem;
            width: 90%;
            max-width: 500px;
            border: 1px solid var(--dark-600);
            box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1);
        }
        
        .modal-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 1.5rem;
            padding-bottom: 1rem;
            border-bottom: 1px solid var(--dark-600);
        }
        
        .modal-title {
            font-size: 1.25rem;
            font-weight: 600;
            color: var(--primary-400);
        }
        
        .modal-close {
            background: none;
            border: none;
            color: var(--dark-300);
            font-size: 1.5rem;
            cursor: pointer;
            padding: 0.5rem;
            border-radius: 0.5rem;
            transition: all 0.3s ease;
        }
        
        .modal-close:hover {
            background: var(--dark-700);
            color: white;
        }
        
        .verification-input {
            width: 100%;
            padding: 0.75rem;
            border: 1px solid var(--dark-600);
            border-radius: 0.5rem;
            background: var(--dark-700);
            color: var(--dark-100);
            font-size: 1.125rem;
            text-align: center;
            letter-spacing: 0.5rem;
            margin-top: 1rem;
        }
        
        .verification-input:focus {
            outline: none;
            border-color: var(--primary-500);
            box-shadow: 0 0 0 3px rgba(168, 85, 247, 0.1);
        }
        
        .operation-description {
            color: var(--dark-300);
            line-height: 1.6;
            margin-bottom: 1rem;
        }
        
        .selected-count {
            background: var(--primary-500);
            color: white;
            padding: 0.25rem 0.5rem;
            border-radius: 0.25rem;
            font-size: 0.875rem;
            margin-right: 0.5rem;
        }
        
        .user-profile {
            display: flex;
            flex-direction: column;
            gap: 1.5rem;
        }
        
        .profile-info {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 1rem;
        }
        
        .info-row {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 0.75rem;
            background: var(--dark-700);
            border-radius: 0.5rem;
            border: 1px solid var(--dark-600);
        }
    </style>
</head>
<body>
    <?php include 'includes/header.php'; ?>
    <?php include 'includes/sidebar.php'; ?>
    
    <!-- Sidebar Overlay for Mobile -->
    <div class="sidebar-overlay" id="sidebarOverlay"></div>

    <!-- Main Content -->
    <div class="main-content" id="mainContent">
        <div class="page-header">
            <h1 class="page-title">👥 مدیریت کاربران</h1>
            <p class="page-description">مدیریت کاربران سیستم و تنظیمات دسترسی</p>
        </div>

        <?php if ($error): ?>
            <div class="alert alert-danger">
                <?php echo htmlspecialchars($error); ?>
            </div>
        <?php endif; ?>

        <?php if ($message): ?>
            <div class="alert alert-success">
                <?php echo htmlspecialchars($message); ?>
            </div>
        <?php endif; ?>

        <!-- Create User Form -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">➕ ایجاد کاربر جدید</h2>
            </div>
            
            <form method="POST" action="" id="createUserForm">
                <input type="hidden" name="action" value="send_create_verification">
                <!-- CSRF token disabled for development -->
                <!-- <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>"> -->
                
                <div class="grid grid-cols-1 grid-cols-2" style="gap: 1rem;">
                    <div class="form-group">
                        <label for="username" class="form-label">نام کاربری *</label>
                        <input type="text" id="username" name="username" class="form-input" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="password" class="form-label">رمز عبور *</label>
                        <input type="password" id="password" name="password" class="form-input" required minlength="8">
                    </div>
                    
                    <div class="form-group">
                        <label for="email" class="form-label">ایمیل</label>
                        <input type="email" id="email" name="email" class="form-input">
                    </div>
                    
                    <div class="form-group" style="display: flex; align-items: end;">
                        <button type="submit" class="btn btn-primary">
                            ارسال کد تأیید
                        </button>
                    </div>
                </div>
            </form>
        </div>

        <!-- Users List with Bulk Operations -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">📋 لیست کاربران</h2>
            </div>
            
            <!-- Bulk Actions -->
            <div class="bulk-actions">
                <button type="button" class="btn-delete" onclick="showBulkOperationModal('delete')" disabled>
                    🗑️ حذف کاربران
                </button>
                <button type="button" class="btn-protect" onclick="showBulkOperationModal('enable_protection')" disabled>
                    🛡️ فعال‌سازی محافظ
                </button>
                <button type="button" class="btn-unprotect" onclick="showBulkOperationModal('disable_protection')" disabled>
                    🔓 غیرفعال‌سازی محافظ
                </button>
                <button type="button" class="btn-activate" onclick="showBulkOperationModal('activate')" disabled>
                    ✅ فعال‌سازی کاربران
                </button>
                <button type="button" class="btn-deactivate" onclick="showBulkOperationModal('deactivate')" disabled>
                    ❌ غیرفعال‌سازی کاربران
                </button>
            </div>
            
            <!-- Users Table -->
            <div class="table-container">
                <table class="users-table">
                    <thead>
                        <tr>
                            <th style="width: 50px;">
                                <input type="checkbox" id="selectAll" class="user-checkbox">
                            </th>
                            <th>نام کاربری</th>
                            <th>ایمیل</th>
                            <th>وضعیت</th>
                            <th>محافظت</th>
                            <th>آخرین ورود</th>
                            <th>تاریخ ایجاد</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($users as $user): ?>
                            <tr>
                                <td>
                                    <input type="checkbox" name="selected_users[]" value="<?php echo $user['id']; ?>" 
                                           class="user-checkbox user-select" 
                                           <?php echo ($user['id'] == $_SESSION['admin_id']) ? 'disabled' : ''; ?>>
                                </td>
                                <td>
                                    <?php echo htmlspecialchars($user['username']); ?>
                                    <?php if ($user['id'] == $_SESSION['admin_id']): ?>
                                        <span class="status-badge status-active">شما</span>
                                    <?php endif; ?>
                                </td>
                                <td><?php echo htmlspecialchars($user['email'] ?? '-'); ?></td>
                                <td>
                                    <?php if ($user['is_active']): ?>
                                        <span class="status-badge status-active">فعال</span>
                                    <?php else: ?>
                                        <span class="status-badge status-inactive">غیرفعال</span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php if ($user['is_protected']): ?>
                                        <span class="status-badge status-protected">🛡️ محافظت شده</span>
                                    <?php else: ?>
                                        <span class="status-badge status-inactive">بدون محافظت</span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php echo $user['last_login'] ? formatDate($user['last_login']) : 'هرگز'; ?>
                                </td>
                                <td><?php echo formatDate($user['created_at']); ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>

        <!-- Current User Profile -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">👤 پروفایل کاربری شما</h2>
            </div>
            
            <?php 
            $currentUser = null;
            foreach ($users as $user) {
                if ($user['id'] == $_SESSION['admin_id']) {
                    $currentUser = $user;
                    break;
                }
            }
            ?>
            
            <?php if ($currentUser): ?>
                <div class="user-profile">
                    <div class="profile-info">
                        <div class="info-row">
                            <strong>نام کاربری:</strong>
                            <span><?php echo htmlspecialchars($currentUser['username']); ?></span>
                        </div>
                        <div class="info-row">
                            <strong>ایمیل:</strong>
                            <span><?php echo htmlspecialchars($currentUser['email'] ?? '-'); ?></span>
                        </div>
                        <div class="info-row">
                            <strong>وضعیت:</strong>
                            <span>
                                <?php if ($currentUser['is_active']): ?>
                                    <span class="status-badge status-active">فعال</span>
                                <?php else: ?>
                                    <span class="status-badge status-inactive">غیرفعال</span>
                                <?php endif; ?>
                            </span>
                        </div>
                        <div class="info-row">
                            <strong>محافظت:</strong>
                            <span>
                                <?php if ($currentUser['is_protected']): ?>
                                    <span class="status-badge status-protected">🛡️ محافظت شده</span>
                                <?php else: ?>
                                    <span class="status-badge status-inactive">بدون محافظت</span>
                                <?php endif; ?>
                            </span>
                        </div>
                    </div>
                    
                    <!-- Update Profile Form -->
                    <?php if (!$currentUser['is_protected']): ?>
                        <form method="POST" action="">
                            <input type="hidden" name="action" value="update">
                            <input type="hidden" name="user_id" value="<?php echo $currentUser['id']; ?>">
                            <!-- CSRF token disabled for development -->
                            <!-- <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>"> -->
                            
                            <div class="form-group">
                                <label for="update_email" class="form-label">تغییر ایمیل</label>
                                <div style="display: flex; gap: 1rem; align-items: end;">
                                    <input type="email" id="update_email" name="email" class="form-input" 
                                           value="<?php echo htmlspecialchars($currentUser['email'] ?? ''); ?>" style="flex: 1;">
                                    <button type="submit" class="btn btn-primary">بروزرسانی</button>
                                </div>
                            </div>
                        </form>
                        
                        <!-- Change Password Form -->
                        <form method="POST" action="" id="changePasswordForm">
                            <input type="hidden" name="action" value="change_password">
                            <input type="hidden" name="user_id" value="<?php echo $currentUser['id']; ?>">
                            <!-- CSRF token disabled for development -->
                            <!-- <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>"> -->
                            
                            <div class="form-group">
                                <label for="new_password" class="form-label">تغییر رمز عبور</label>
                                <div style="display: flex; gap: 1rem; align-items: end;">
                                    <input type="password" id="new_password" name="new_password" class="form-input" 
                                           placeholder="رمز عبور جدید (حداقل 8 کاراکتر)" minlength="8" style="flex: 1;">
                                    <button type="submit" class="btn btn-warning">تغییر رمز</button>
                                </div>
                            </div>
                        </form>
                    <?php else: ?>
                        <div class="alert alert-warning">
                            <strong>⚠️ محدودیت محافظت:</strong> کاربران محافظت شده نمی‌توانند اطلاعات خود را تغییر دهند.
                        </div>
                    <?php endif; ?>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- Create User Modal -->
    <div id="createUserModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3 class="modal-title">ایجاد کاربر جدید</h3>
                <button class="modal-close" onclick="closeCreateModal()">&times;</button>
            </div>
            <div class="modal-body">
                <div class="operation-description">
                    کد تأیید به ایمیل شما ارسال شد. لطفاً آن را وارد کنید.
                </div>
                <input type="text" id="createVerificationCode" class="verification-input" 
                       placeholder="000000" maxlength="6" pattern="[0-9]{6}">
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" onclick="closeCreateModal()">انصراف</button>
                <button type="button" class="btn btn-primary" onclick="executeCreateUser()">ایجاد کاربر</button>
            </div>
        </div>
    </div>

    <!-- Bulk Operation Modal -->
    <div id="bulkOperationModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3 class="modal-title" id="modalTitle">عملیات گروهی</h3>
                <button class="modal-close" onclick="closeModal()">&times;</button>
            </div>
            <div class="modal-body">
                <div class="operation-description" id="operationDescription">
                    لطفاً کد تأیید ارسال شده به ایمیل خود را وارد کنید.
                </div>
                <input type="text" id="verificationCode" class="verification-input" 
                       placeholder="000000" maxlength="6" pattern="[0-9]{6}">
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" onclick="closeModal()">انصراف</button>
                <button type="button" class="btn btn-primary" onclick="executeBulkOperation()">تأیید و اجرا</button>
            </div>
        </div>
    </div>

    <script>
        let currentOperation = '';
        let selectedUsers = [];
        
        // Toast notification function
        function showToast(message, type = 'success') {
            // Remove existing toasts
            const existingToasts = document.querySelectorAll('.toast');
            existingToasts.forEach(toast => toast.remove());
            
            // Create new toast
            const toast = document.createElement('div');
            toast.className = `toast toast-${type}`;
            toast.textContent = message;
            
            document.body.appendChild(toast);
            
            // Show toast
            setTimeout(() => {
                toast.classList.add('show');
            }, 100);
            
            // Hide toast after 3 seconds
            setTimeout(() => {
                toast.classList.remove('show');
                setTimeout(() => {
                    if (toast.parentNode) {
                        toast.parentNode.removeChild(toast);
                    }
                }, 300);
            }, 3000);
        }
        
        // Create user form submission
        document.getElementById('createUserForm').addEventListener('submit', function(e) {
            e.preventDefault();
            
            const formData = new FormData(this);
            
            fetch('users.php', {
                method: 'POST',
                body: formData,
                headers: {
                    'X-Requested-With': 'XMLHttpRequest'
                }
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // Show create user modal
                    document.getElementById('createUserModal').style.display = 'block';
                    document.getElementById('createVerificationCode').focus();
                    showToast(data.message, 'success');
                } else {
                    showToast(data.error || 'خطا در ارسال کد تأیید', 'error');
                }
            })
            .catch(error => {
                console.error('Error sending verification code:', error);
                showToast('خطا در ارسال کد تأیید', 'error');
            });
        });
        
        function closeCreateModal() {
            document.getElementById('createUserModal').style.display = 'none';
            document.getElementById('createVerificationCode').value = '';
        }
        
        function executeCreateUser() {
            const verificationCode = document.getElementById('createVerificationCode').value;
            
            if (!verificationCode || verificationCode.length !== 6) {
                showToast('لطفاً کد تأیید 6 رقمی را وارد کنید', 'warning');
                return;
            }
            
            const formData = new FormData();
            formData.append('action', 'create_user');
            formData.append('verification_code', verificationCode);
            // formData.append('csrf_token', '<?php echo generateCSRFToken(); ?>'); // Disabled for development
            
            fetch('users.php', {
                method: 'POST',
                body: formData,
                headers: {
                    'X-Requested-With': 'XMLHttpRequest'
                }
            })
            .then(response => response.json())
            .then(data => {
                closeCreateModal();
                if (data.success) {
                    showToast(data.message, 'success');
                    setTimeout(() => location.reload(), 1000);
                } else {
                    showToast(data.error || 'خطا در ایجاد کاربر', 'error');
                }
            })
            .catch(error => {
                console.error('Error creating user:', error);
                showToast('خطا در ایجاد کاربر', 'error');
            });
        }
        
        // Select all functionality
        document.getElementById('selectAll').addEventListener('change', function() {
            const checkboxes = document.querySelectorAll('.user-select:not(:disabled)');
            checkboxes.forEach(checkbox => {
                checkbox.checked = this.checked;
            });
            updateBulkActions();
        });
        
        // Individual checkbox change
        document.querySelectorAll('.user-select').forEach(checkbox => {
            checkbox.addEventListener('change', updateBulkActions);
        });
        
        function updateBulkActions() {
            const checkboxes = document.querySelectorAll('.user-select:checked');
            const bulkButtons = document.querySelectorAll('.bulk-actions button');
            
            selectedUsers = Array.from(checkboxes).map(cb => cb.value);
            
            bulkButtons.forEach(button => {
                button.disabled = selectedUsers.length === 0;
            });
        }
        
        function showBulkOperationModal(operation) {
            currentOperation = operation;
            const modal = document.getElementById('bulkOperationModal');
            const title = document.getElementById('modalTitle');
            const description = document.getElementById('operationDescription');
            
            const operations = {
                'delete': {
                    title: 'حذف کاربران',
                    description: `آیا از حذف <span class="selected-count">${selectedUsers.length}</span> کاربر انتخاب شده اطمینان دارید؟ این عملیات قابل بازگشت نیست.`
                },
                'enable_protection': {
                    title: 'فعال‌سازی محافظ',
                    description: `آیا از فعال‌سازی محافظ برای <span class="selected-count">${selectedUsers.length}</span> کاربر انتخاب شده اطمینان دارید؟`
                },
                'disable_protection': {
                    title: 'غیرفعال‌سازی محافظ',
                    description: `آیا از غیرفعال‌سازی محافظ برای <span class="selected-count">${selectedUsers.length}</span> کاربر انتخاب شده اطمینان دارید؟`
                },
                'activate': {
                    title: 'فعال‌سازی کاربران',
                    description: `آیا از فعال‌سازی <span class="selected-count">${selectedUsers.length}</span> کاربر انتخاب شده اطمینان دارید؟`
                },
                'deactivate': {
                    title: 'غیرفعال‌سازی کاربران',
                    description: `آیا از غیرفعال‌سازی <span class="selected-count">${selectedUsers.length}</span> کاربر انتخاب شده اطمینان دارید؟`
                }
            };
            
            title.textContent = operations[operation].title;
            description.innerHTML = operations[operation].description;
            
            // Send verification code
            sendVerificationCode(operation);
            
            modal.style.display = 'block';
            document.getElementById('verificationCode').focus();
        }
        
        function closeModal() {
            document.getElementById('bulkOperationModal').style.display = 'none';
            document.getElementById('verificationCode').value = '';
        }
        
        function sendVerificationCode(operation) {
            const formData = new FormData();
            formData.append('action', 'send_verification_code');
            formData.append('operation', operation);
            // formData.append('csrf_token', '<?php echo generateCSRFToken(); ?>'); // Disabled for development
            selectedUsers.forEach(userId => {
                formData.append('selected_users[]', userId);
            });
            
            fetch('users.php', {
                method: 'POST',
                body: formData,
                headers: {
                    'X-Requested-With': 'XMLHttpRequest'
                }
            })
            .then(response => response.json())
            .then(data => {

                if (data.success) {
                    // Store the operation ID for later use
                    window.currentOperationId = data.operation_id;
                    showToast(data.message, 'success');
                } else {
                    showToast('خطا در ارسال کد تأیید', 'error');
                }
            })
            .catch(error => {
                console.error('Error sending verification code:', error);
                showToast('خطا در ارسال کد تأیید', 'error');
            });
        }
        
        function executeBulkOperation() {
            const verificationCode = document.getElementById('verificationCode').value;
            
            if (!verificationCode || verificationCode.length !== 6) {
                showToast('لطفاً کد تأیید 6 رقمی را وارد کنید', 'warning');
                return;
            }
            
            const formData = new FormData();
            formData.append('action', 'execute_bulk_operation');
            formData.append('operation', currentOperation);
            formData.append('verification_code', verificationCode);
            // formData.append('csrf_token', '<?php echo generateCSRFToken(); ?>'); // Disabled for development
            // Get operation_id from the session that was set when verification code was sent
            formData.append('operation_id', window.currentOperationId || '');
            
            fetch('users.php', {
                method: 'POST',
                body: formData,
                headers: {
                    'X-Requested-With': 'XMLHttpRequest'
                }
            })
            .then(response => response.json())
            .then(data => {
                closeModal();
                if (data.success) {
                    showToast(data.message, 'success');
                    setTimeout(() => location.reload(), 1000);
                } else {
                    showToast(data.error || 'خطا در اجرای عملیات', 'error');
                }
            })
            .catch(error => {
                console.error('Error executing operation:', error);
                showToast('خطا در اجرای عملیات', 'error');
            });
        }
        
        // Close modals when clicking outside
        window.onclick = function(event) {
            const createModal = document.getElementById('createUserModal');
            const bulkModal = document.getElementById('bulkOperationModal');
            
            if (event.target === createModal) {
                closeCreateModal();
            }
            if (event.target === bulkModal) {
                closeModal();
            }
        }
        
        // Enter key to submit verification codes
        document.getElementById('createVerificationCode').addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                executeCreateUser();
            }
        });
        
        document.getElementById('verificationCode').addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                executeBulkOperation();
            }
        });
        
        // Initialize bulk actions state
        updateBulkActions();
    </script>
</body>
</html>